<?php

namespace Comitium5\PaymentBundle\Payments\Stripe;

use Comitium5\PaymentBundle\Components\Interfaces\PaymentInterface;
use Comitium5\PaymentBundle\Components\Interfaces\TransactionInterface;
use Comitium5\PaymentBundle\Components\Validators\RedSys\StripeServiceValidator;
use Comitium5\PaymentBundle\Exception\Stripe\NoApiKeyProvidedException;
use \Stripe\Stripe;
use \Stripe\Charge;

/**
 * Class StripeService
 *
 * @author Daniel J. Pérez <danieljordi@bab-soft.com>
 * @package Comitium5\PaymentBundle\Payments\Stripe
 */
class StripeService implements PaymentInterface
{
    /**
     * @var string
     */
    protected $privateKey;

    /**
     * StripeService constructor.
     *
     * @param string $privateKey
     */
    public function __construct($privateKey)
    {
        $this->privateKey = $privateKey;
    }

    /**
     * @param TransactionInterface $transaction
     *
     * @throws NoApiKeyProvidedException
     * @return mixed
     */
    public function pay(TransactionInterface $transaction)
    {
        StripeServiceValidator::validate($this->privateKey);

        Stripe::setApiKey($this->privateKey);
        $card = $transaction->getCard();

        try {

            $chargeData = [
                'amount'   => $transaction->getAmount(),
                'currency' => $transaction->getCurrency(),
                'source'   => $card->getToken(),
            ];

            $metadata = $transaction->getMetadata();
            $customer = $transaction->getCustomer();

            if (!empty($metadata)) {
                $chargeData['metadata'] = $metadata;
            }

            if (!empty($customer)) {
                $chargeData['customer'] = $customer->id;
            }

            $charge = Charge::create($chargeData);

            return $charge;

        } catch (\Throwable $e) {
            return false;
        }
    }

    /**
     * @return string
     */
    public function getPrivateKey()
    {
        return $this->privateKey;
    }

    /**
     * @param string $privateKey
     *
     * @return $this
     */
    public function setPrivateKey($privateKey)
    {
        $this->privateKey = $privateKey;

        return $this;
    }
}
