<?php

namespace Tests\Payments\RedSys;

use Comitium5\PaymentBundle\Components\Client\GuzzleClient;
use Comitium5\PaymentBundle\Components\Constants\RedSys\RedSysConfig;
use Comitium5\PaymentBundle\Components\Constants\RedSys\RedSysParameters;
use Comitium5\PaymentBundle\Components\Factories\Client\RedSysGuzzleFactory;
use Comitium5\PaymentBundle\Components\Interfaces\TransactionInterface;
use Comitium5\PaymentBundle\Components\Model\RedSys\ConcurrentTransaction;
use Comitium5\PaymentBundle\Components\Model\RedSys\Transaction;
use Comitium5\PaymentBundle\Components\Model\Stripe\Transaction as StripeTransaction;
use Comitium5\PaymentBundle\Components\ValueObjects\PaymentObject;
use Comitium5\PaymentBundle\Payments\Client\RedSysHttpClientService;
use Comitium5\PaymentBundle\Payments\RedSys\RedSysBackgroundPaymentService;
use Comitium5\PaymentBundle\Payments\RedSys\RedSysValidationService;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\ResponseInterface;
use Tests\UnitTestCase;

/**
 * Class RedSysBackgroundPaymentServiceTest
 *
 * @author Carles Gómez <carles@bab-soft.com>
 * @package Tests\Payments\RedSys
 */
class RedSysBackgroundPaymentServiceTest extends UnitTestCase
{
    /**
     * @dataProvider dataTransactions
     *
     * @param TransactionInterface $transaction
     *
     * @throws \Exception
     */
    public function testPay(TransactionInterface $transaction)
    {
        $container = [];

        $guzzleClient = new GuzzleClient($this->client($container));

        $service = new RedSysBackgroundPaymentService(
            new RedSysHttpClientService($guzzleClient),
            RedSysConfig::SERVICE_PAY_ENDPOINT,
            base64_encode("123456789012345678901234"),
            "bar"
        );

        $paymentObject = $service->pay($transaction);

        $this->assertInstanceOf(PaymentObject::class, $paymentObject);

        foreach ($container as $key => $transaction) {
            /**
             * @var $request Request
             */
            $request = $transaction['request'];

            $params = $request->getBody()->getContents();

            $this->assertEquals('https', $request->getUri()->getScheme());
            $this->assertEquals('sis.redsys.es', $request->getHeader('Host')[0]);
            $this->assertEquals('/sis/realizarPago', $request->getUri()->getPath());
            $this->assertContains("Ds_MerchantParameters", $params);
            $this->assertContains("Ds_Signature", $params);
            $this->assertContains("Ds_SignatureVersion", $params);
            $this->assertEquals("POST", $request->getMethod());
        }
    }

    /**
     * @return array
     */
    public function dataTransactions()
    {
        return [
            [$this->transaction()],
            [$this->concurrentTransaction()],
        ];
    }

    /**
     * @dataProvider sandboxData
     *
     * @param TransactionInterface $transaction
     * @param string               $privateKey
     *
     * @throws \Exception
     */
    public function testSandboxPay(TransactionInterface $transaction, $privateKey)
    {
        $service = new RedSysBackgroundPaymentService(
            (new RedSysGuzzleFactory())->createForTest(),
            RedSysConfig::SERVICE_PAY_REST_TEST,
            $privateKey,
            'foo'
        );

        $paymentObject = $service->pay($transaction);

        $this->assertInstanceOf(PaymentObject::class, $paymentObject);
        $this->assertInstanceOf(ResponseInterface::class, $paymentObject->getData());
        $this->assertGreaterThanOrEqual("200", $paymentObject->getData()->getStatusCode());
        $this->assertLessThan("300", $paymentObject->getData()->getStatusCode());

        $response = \json_decode($paymentObject->getData()->getBody(), true);

        $this->assertArrayHasKey(RedSysParameters::SIGNATURE, $response);
        $this->assertArrayHasKey(RedSysParameters::MERCHANT_PARAMETERS, $response);

        $validation = new RedSysValidationService($privateKey);
        $response = $validation->validate(
            $response[RedSysParameters::MERCHANT_PARAMETERS],
            $response[RedSysParameters::SIGNATURE]
        );

        $this->assertTrue($response->isValid());
    }

    /**
     * @return array
     */
    public function sandboxData()
    {
        return [
            [$this->sandboxConcurrentTransaction(), 'sq7HjrUOBfKmC576ILgskD5srU870gJ7'],
        ];
    }

    /**
     * @dataProvider validationData
     *
     * @param $privateKey
     * @param $fuc
     * @param TransactionInterface $transaction
     * @param $expectedExceptionMessage
     *
     * @throws \Exception
     */
    public function testPayOnValidationFail($privateKey, $fuc, TransactionInterface $transaction, $expectedExceptionMessage)
    {
        $service = new RedSysBackgroundPaymentService(
            $this->createHttpClientService(),
            RedSysConfig::SERVICE_PAY_ENDPOINT,
            $privateKey,
            $fuc
        );

        $this->expectExceptionMessage($expectedExceptionMessage);

        $service->pay($transaction);
    }

    /**
     * @return array
     */
    public function validationData()
    {
        return [
            [
                null,
                "foo",
                $this->stripeTransaction(),
                "Transaction must be instance of RedSysTransactionInterface"
            ],
            [
                "",
                "foo",
                $this->concurrentTransaction(),
                "Missing private key"
            ],
            [
                "foo",
                false,
                $this->concurrentTransaction(),
                "Missing FUC value"
            ],
        ];
    }

    /**
     * @return \Comitium5\PaymentBundle\Payments\Client\RedSysHttpClientService
     */
    private function createHttpClientService()
    {
        $factory = new RedSysGuzzleFactory();

        return $factory->create();
    }

    /**
     * @param array $container
     *
     * @return Client
     */
    private function client(array &$container)
    {
        $mock = new MockHandler([
            new Response(200)
        ]);

        $history = Middleware::history($container);

        $stack = HandlerStack::create($mock);

        $stack->push($history);

        return new Client([
            'handler' => $stack,
            'base_uri' => "https://www.foo.com/",
        ]);
    }

    /**
     * @return Transaction
     */
    private function transaction()
    {
        return new Transaction(
            300,
            "Product description",
            "Foo",
            "999008881",
            "www.foo.es",
            "www.foo.es/ok",
            "www.foo.es/ko",
            "Bar",
            [
                "data1" => "foo",
                "data2" => "bar",
            ]
        );
    }

    /**
     * @return Transaction
     */
    private function concurrentTransaction()
    {
        return new ConcurrentTransaction(
            300,
            "Product description",
            "Foo",
            "999008881",
            "www.foo.es",
            "www.foo.es/ok",
            "www.foo.es/ko",
            "Bar",
            [
                "data1" => "foo",
                "data2" => "bar",
            ]
        );
    }

    /**
     * @return ConcurrentTransaction
     */
    private function sandboxConcurrentTransaction()
    {
        return new ConcurrentTransaction(
            300,
            "Product description",
            "Foo",
            "66554221",
            "www.foo.es",
            "www.foo.es/ok",
            "www.foo.es/ko",
            "Bar",
            [
                "data1" => "foo",
                "data2" => "bar",
            ],
            time(),
            '520f4dcc7cdba53e319517d7e39f0879312ca3fb'
        );
    }

    /**
     * @return StripeTransaction
     */
    private function stripeTransaction()
    {
        return new StripeTransaction();
    }
}
