<?php

namespace Tests\Components\Helpers\RedSys;

use Comitium5\PaymentBundle\Components\Constants\RedSys\RedSysParameters;
use Comitium5\PaymentBundle\Components\Helpers\RedSys\Encoder;
use Comitium5\PaymentBundle\Components\Interfaces\TransactionInterface;
use Comitium5\PaymentBundle\Components\Model\RedSys\ConcurrentTransaction;
use Comitium5\PaymentBundle\Components\Model\RedSys\Transaction;
use Tests\UnitTestCase;

/**
 * Class EncoderTest
 *
 * @author Carles Gómez <carles@bab-soft.com>
 * @package Tests\Components\Helpers\RedSys
 */
class EncoderTest extends UnitTestCase
{
    /**
     *
     */
    public function testEncodeParamsForSignature()
    {
        $transaction = $this->transaction();

        $encodedParams = Encoder::encodeParamsForSignature($transaction);

        $decodedParams = base64_decode($encodedParams);

        $params = json_decode($decodedParams, true);

        $this->assertNotFalse($decodedParams);

        $this->assertTransactionParamKeys($params);
        $this->assertTransactionValues($transaction, $params);

        $this->assertArrayNotHasKey("Ds_Merchant_Identifier", $params);
        $this->assertArrayNotHasKey("Ds_Merchant_DirectPayment", $params);
    }

    /**
     *
     */
    public function testEncodeParamsForConcurrentSignature()
    {
        $concurrentTransaction = $this->concurrentTransaction();

        $encodedParams = Encoder::encodeParamsForSignature($concurrentTransaction);

        $decodedParams = base64_decode($encodedParams);

        $params = json_decode($decodedParams, true);

        $this->assertNotFalse($decodedParams);

        $this->assertConcurrentTransactionParamKeys($params);
        $this->assertTransactionValues($concurrentTransaction, $params);

        $this->assertArrayHasKeys([
            "Ds_Merchant_Identifier",
            "Ds_Merchant_DirectPayment",
        ], $params);
    }

    /**
     * @param TransactionInterface $transaction
     * @param array $params
     */
    private function assertTransactionValues(TransactionInterface $transaction, array $params)
    {
        $this->assertEquals($transaction->getAmount() * 100, $params["Ds_Merchant_Amount"]);

        $privateData = json_decode(base64_decode($params["Ds_Merchant_MerchantData"]), true);

        $this->assertEquals([
            "data1" => "foo",
            "data2" => "bar",
        ], $privateData);
    }

    /**
     * @param array $params
     */
    private function assertGeneralTransactionParamKeys(array $params)
    {
        $this->assertArrayHasKeys([
            RedSysParameters::MERCHANT_AMOUNT,
            RedSysParameters::MERCHANT_CURRENCY,
            RedSysParameters::MERCHANT_ORDER,
            RedSysParameters::MERCHANT_PRODUCT_DESCRIPTION,
            RedSysParameters::MERCHANT_TITULAR,
            RedSysParameters::MERCHANT_CODE,
            RedSysParameters::MERCHANT_NAME,
            RedSysParameters::MERCHANT_CONSUMER_LANGUAGE,
            RedSysParameters::MERCHANT_TERMINAL,
            RedSysParameters::MERCHANT_TRANSACTION_TYPE,
            RedSysParameters::MERCHANT_MERCHANT_DATA,
        ], $params);
    }

    /**
     * @param array $params
     */
    private function assertTransactionParamKeys(array $params)
    {
        $this->assertGeneralTransactionParamKeys($params);

        $this->assertArrayHasKeys([
            RedSysParameters::MERCHANT_URL,
            RedSysParameters::MERCHANT_URL_OK,
            RedSysParameters::MERCHANT_URL_KO,
            RedSysParameters::MERCHANT_PAY_METHODS,
        ], $params);
    }

    /**
     * @param array $params
     */
    private function assertConcurrentTransactionParamKeys(array $params)
    {
        $this->assertGeneralTransactionParamKeys($params);

        $this->assertArrayHasKeys([
            RedSysParameters::MERCHANT_IDENTIFIER,
            RedSysParameters::MERCHANT_DIRECT_PAYMENT,
            RedSysParameters::MERCHANT_EXCEPSCA,
        ], $params);
    }

    /**
     * @return Transaction
     */
    private function transaction()
    {
        return new Transaction(
            300,
            "Product description",
            "Foo",
            "999008881",
            "www.foo.es",
            "www.foo.es/ok",
            "www.foo.es/ko",
            "Bar",
            [
                "data1" => "foo",
                "data2" => "bar",
            ]
        );
    }

    /**
     * @return Transaction
     */
    private function concurrentTransaction()
    {
        return new ConcurrentTransaction(
            300,
            "Product description",
            "Foo",
            "999008881",
            "www.foo.es",
            "www.foo.es/ok",
            "www.foo.es/ko",
            "Bar",
            [
                "data1" => "foo",
                "data2" => "bar",
            ]
        );
    }
}
