<?php

namespace Comitium5\MailingBundle\Command;

use Comitium5\MailingBundle\Model\Campaign\CampaignInterface;
use Comitium5\MailingBundle\Services\Dispatcher\CampaignDispatcher;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class CampaignSenderCommand
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\MailingBundle\Command
 */
class CampaignSenderCommand extends ContainerAwareCommand
{
    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this
            ->setName('comitium5_mailing:campaign:send')
            ->setDescription('Send campaigns')
            ->addOption("host", null, InputOption::VALUE_OPTIONAL, "Host to set on content links")
            ->addOption("campaignsId", null, InputOption::VALUE_OPTIONAL, "Single or multiple campaigns id separated by commas")
            ->addOption("states", null, InputOption::VALUE_OPTIONAL, "Single or multiple states separated by commas. By default only ready campaigns will be process", CampaignInterface::STATE_SENDING)
            ->addOption("onlyPastStartDate", null, InputOption::VALUE_OPTIONAL, "Get only campaigns with start date before than now. By default true", true)
            ->addOption("limit", null, InputOption::VALUE_OPTIONAL, "Limit of emails to send", false)
            ->addOption("clearList", null, InputOption::VALUE_OPTIONAL, "Clear list after send", true)
            ->addOption("threads", null, InputOption::VALUE_OPTIONAL, "Number of parallel threads", 16)
            ->addOption("extraDataFilter", null, InputOption::VALUE_OPTIONAL, "JSON of extra filters")
        ;
    }

    /**
     * @param array $states
     * @return array
     * @throws \Exception
     */
    private function checkValidStates(array $states)
    {
        $validStates = array(
            CampaignInterface::STATE_DRAFT,
            CampaignInterface::STATE_READY,
            CampaignInterface::STATE_SENDING,
            CampaignInterface::STATE_SENT,
        );

        foreach ($states as $state) {
            if (!in_array($state, $validStates)) {
                throw new \Exception("Invalid state value {$state}. Set one of these valid states: ".implode(",", $validStates));
            }
        }

        return $states;
    }


    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @throws \Exception
     * @return void
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $host = $input->getOption("host");
        $campaignsId = $input->getOption("campaignsId") !== null ? explode(",",$input->getOption("campaignId")) : array();
        $states = $this->checkValidStates(explode(",",$input->getOption("states")));
        $onlyPastStartDate = $input->getOption("onlyPastStartDate");
        $limit = $input->getOption("limit");
        $clearList = $input->getOption("clearList");
        $threads = $input->getOption("threads");
        $extraFilters = json_decode($input->getOption("extraDataFilter"), true);

        if ($host !== null && filter_var($host, FILTER_VALIDATE_URL) === false) {
            throw new \Exception("Host option must be a valid url like http://foo.com");
        }

        $container = $this->getContainer();
        $campaignManager = $container->get("comitium5_mailing.manager.campaign");
        $campaignRepository = $campaignManager->getRepository();
        $campaignDispatcher = $container->get("comitium5_mailing.dispatcher.campaign");

        $campaignDispatcher
            ->setRootDir($container->getParameter("kernel.root_dir"))
            ->setThreads($threads)
            ->setHost($host)
        ;

        if ($limit !== false && is_numeric($limit)) {
            $campaignDispatcher->setLimit($limit);
        }

        $campaignDispatcher->setEnvironment($container->getParameter('kernel.environment'));

        if (!$clearList) {
            $campaignDispatcher->setRemoveMailingListAfterSent(false);
        }

        $builder = $campaignRepository
            ->createQueryBuilder("c")
            ->where("c.state IN (:states)")
            ->orderBy("c.createdDate", "ASC")
            ->setParameter("states", $states);

        if ($onlyPastStartDate === true) {
            $builder
                ->andWhere("c.startDate <= :now")
                ->setParameter("now", new \DateTime());
        }

        if (!empty($campaignsId)) {
            $builder
                ->andWhere("c.id IN (:ids)")
                ->setParameter("ids", $campaignsId);
        }

        $campaigns = $builder
            ->getQuery()
            ->getResult()
        ;

        $campaignsCount = count($campaigns);

        foreach ($campaigns as $key => $campaign) {
            if (!empty($extraFilters)) {
                $campaignData = $campaign->getExtraData();
                if ($this->ignoreCampaignByFilter($campaignData, $extraFilters)) {
                    $campaignsCount--;
                    continue;
                }
            }

            $campaignResponse = $campaignDispatcher->dispatch($campaign);

            $output->writeln("Camapaign {$campaignResponse->getCampaign()->getId()}: Senders {$campaignResponse->getResult()}", OutputInterface::VERBOSITY_VERBOSE);
        }

        if ($campaignsCount === 0) {
            $output->writeln("Nothing to do", OutputInterface::VERBOSITY_VERBOSE);
        }
    }

    /**
     * @param $campaignData
     * @param $filters
     *
     * @return bool
     */
    private function ignoreCampaignByFilter($campaignData, $filters)
    {
        $intersect = array_intersect_assoc($campaignData, $filters);

        return count($intersect) !== count($filters);
    }
}
