<?php

namespace Comitium5\MailingBundle\Services\Parser;

use Comitium5\CommonWidgetsBundle\MailingStats\StatsUtils;
use Comitium5\MailingBundle\Model\MailingList\MailingListInterface;
use Symfony\Bundle\FrameworkBundle\Routing\Router;
use Symfony\Component\DomCrawler\Crawler;

/**
 * Class CampaignContentParser
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\MailingBundle\Services\Parser
 */
class CampaignContentParser
{
    const ENCODE_PARAM = "_d";
    const REDIRECT_PARAM = "_r";
    const CAMPAIGN_ID_PARAM = "_cid";
    const EMAIL_PARAM = "_e";
    const EXTRA_DATA_PARAM = "_ext";

    /**
     * @var Router
     */
    protected $router;

    /**
     * @var string
     */
    protected $host;

    /**
     * @var string
     */
    protected $scheme;

    /**
     * @var string
     */
    protected $baseUrl;

    /**
     * CampaignContentParser constructor.
     * @param Router $router
     * @param string $host
     * @param string $scheme
     * @param string $baseUrl
     */
    public function __construct(
        Router $router,
        $host = null,
        $scheme = null,
        $baseUrl = null
    ) {
        $this->router  = $router;
        $this->host    = $host;
        $this->scheme  = $scheme;
        $this->baseUrl = $baseUrl;
    }

    /**
     * @return Router
     */
    public function getRouter()
    {
        return $this->router;
    }

    /**
     * @param Router $router
     * @return $this
     */
    public function setRouter($router)
    {
        $this->router = $router;
        return $this;
    }

    /**
     * @return string
     */
    public function getHost()
    {
        return $this->host;
    }

    /**
     * @param string $host
     * @return $this
     */
    public function setHost($host)
    {
        $this->host = $host;

        return $this;
    }

    /**
     * @return string
     */
    public function getScheme()
    {
        return $this->scheme;
    }

    /**
     * @param string $scheme
     * @return $this
     */
    public function setScheme($scheme)
    {
        $this->scheme = $scheme;

        return $this;
    }

    /**
     * @return string
     */
    public function getBaseUrl()
    {
        return $this->baseUrl;
    }

    /**
     * @param string $baseUrl
     * @return $this
     */
    public function setBaseUrl($baseUrl)
    {
        $this->baseUrl = $baseUrl;

        return $this;
    }

    /**
     * @param MailingListInterface $mailingList
     * @param $html
     *
     * @return mixed
     */
    public function parse(MailingListInterface $mailingList, $html)
    {
        $this->buildRouterContext();

        return $this->generateHtml($mailingList, $html);
    }

    /**
     * @param array $data
     * @return string
     */
    public static function encode(array $data)
    {
        return base64_encode(json_encode($data));
    }

    /**
     * @param $data
     * @return mixed
     */
    public static function decode($data)
    {
        $data = base64_decode($data);

        return json_decode($data,true);
    }

    /**
     * @return $this
     */
    private function buildRouterContext()
    {
        if ($this->host !== null) {
            $this
                ->router
                ->getContext()
                ->setHost($this->host);
        }

        if ($this->scheme !== null) {
            $this
                ->router
                ->getContext()
                ->setScheme($this->scheme);
        }

        if ($this->baseUrl !== null) {
            $this
                ->router
                ->getContext()
                ->setBaseUrl($this->baseUrl);
        }

        return $this;
    }

    /**
     * @param MailingListInterface $mailingList
     * @param Crawler $crawler
     * @return string
     */
    private function clickStats(MailingListInterface $mailingList, Crawler $crawler)
    {
        $crawler->filter("a")->each(function (Crawler $crawler) use ($mailingList) {
            $node = $crawler->getNode(0);
            $href = $node->getAttribute("href");
            $url = $this
                ->router
                ->generate(
                    "comitium5_common_widgets_stats_compute_click",
                    [
                        StatsUtils::ENCODE_PARAM => StatsUtils::encode(
                            [
                                StatsUtils::REDIRECT_PARAM    => $href,
                                StatsUtils::CAMPAIGN_ID_PARAM => $mailingList->getCampaign()->getId(),
                                StatsUtils::EMAIL_PARAM       => $mailingList->getEmail(),
                                StatsUtils::EXTRA_DATA_PARAM  => $href,
                            ]
                        )
                    ],
                    Router::ABSOLUTE_URL
                )
            ;

            $node->setAttribute("href", $url);
        });

        $html = '';

        foreach ($crawler as $domElement) {
            $html .= $domElement->ownerDocument->saveHTML($domElement);
        }

        return $html;
    }

    /**
     * @param MailingListInterface $mailingList
     * @param $content
     *
     * @return string|string[]|null
     */
    private function openStats(MailingListInterface $mailingList, $content)
    {
        if (preg_match("/<\/body>/", $content) === 1) {
            $url = $this
                ->router
                ->generate(
                    "comitium5_common_widgets_stats_compute_open",
                    [
                        StatsUtils::ENCODE_PARAM => StatsUtils::encode(
                            [
                                StatsUtils::CAMPAIGN_ID_PARAM => $mailingList->getCampaign()->getId(),
                                StatsUtils::EMAIL_PARAM       => $mailingList->getEmail(),
                                StatsUtils::EXTRA_DATA_PARAM  => "",
                            ]
                        )
                    ],
                    Router::ABSOLUTE_URL
                )
            ;

            $element = '<img src="'.$url.'" width="1px" height="1px"/>';
            $content = preg_replace("/<\/body>/", "{$element}\n</body>", $content);
        }

        return $content;
    }

    /**
     * @param MailingListInterface $mailingList
     * @param $html
     * @return mixed
     */
    private function generateHtml(MailingListInterface $mailingList, $html)
    {
        $html = preg_replace(
            "/ISO-8859-1/i",
            "utf-8",
            $html
        );

        $crawler = new Crawler($html);

        $this->clickStats($mailingList, $crawler);

        $html = $this->openStats(
            $mailingList,
            $crawler->html()
        );

        return $this->wrapHtml($html);
    }

    /**
     * @param $html
     *
     * @return string
     */
    private function wrapHtml($html)
    {
        return sprintf("<!DOCTYPE html><html>%s</html>", $html);
    }
}