<?php

namespace Comitium5\MailingBundle\Command;

use Comitium5\MailingBundle\Entity\BouncedList;
use Comitium5\MailingBundle\Entity\MailingList;
use Comitium5\MailingBundle\Event\BouncedEmailEvent;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class CampaignSenderCommand
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\MailingBundle\Command
 */
class SendThreadCommand extends ContainerAwareCommand
{
    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this
            ->setName('comitium5_mailing:campaign:send-thread')
            ->setDescription('Send campaigns')
            ->addOption("thread", null, InputOption::VALUE_REQUIRED)
            ->addOption("data", null, InputOption::VALUE_REQUIRED)
            ->addOption("delete", null, InputOption::VALUE_REQUIRED, "Delete mailing list item after email is sended", true)
            ->addOption("host", null, InputOption::VALUE_REQUIRED, "Host to set on links")
        ;
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @throws \Exception
     * @return void
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $data   = json_decode($input->getOption("data"), true);
        $thread = $input->getOption("thread");
        $delete = $input->getOption("delete");
        $host   = $input->getOption("host");

        $container              = $this->getContainer();
        $campaignDispatcher     = $container->get("comitium5_mailing.dispatcher.campaign");
        $campaignDispatcher->setRootDir($container->getParameter("kernel.root_dir"));

        if ($host !== null) {
            $urlParsed = parse_url($host);

            $campaignDispatcher
                ->getBuilder()
                ->getParser()
                ->setHost($urlParsed["host"])
                ->setScheme($urlParsed["scheme"])
                ->setBaseUrl(isset($urlParsed["path"]) ? $urlParsed["path"] : null)
            ;
        }

        foreach ($data as $item) {

            $mailingList = $this->findMailingList($item["campaignId"], $item["email"]);

            if (!$mailingList) {
                continue;
            }

            try {
                $response = $campaignDispatcher->dispatchMailingList($mailingList, $thread);

                $result = $response->getResult();

                if ($result === 1 && $delete) {
                    // In case of success
                    $this->delete($mailingList);
                } elseif ($result !== 1) {
                    // In case of any kind of error
                    $this->handleError($mailingList, sprintf("Service return %s results instead 1", $result));
                }
            } catch (\Exception $e) {
                $this->handleError($mailingList, $e->getMessage());
            }
        }
    }

    /**
     * @param $campaignId
     * @param $email
     * @return MailingList|null
     */
    private function findMailingList($campaignId, $email)
    {
        return $this
            ->getContainer()
            ->get("comitium5_mailing.repository.mailing_list")
            ->findOneBy([
                "campaign" => $campaignId,
                "email"    => $email
            ]);
    }

    /**
     * @param MailingList $mailingList
     * @return mixed
     */
    private function delete(MailingList $mailingList)
    {
        return $this
            ->getContainer()
            ->get("comitium5_mailing.repository.mailing_list")
            ->remove($mailingList);
    }

    /**
     * @param MailingList $mailingList
     * @param             $message
     */
    private function handleError(MailingList $mailingList, $message)
    {
        $campaign = $mailingList->getCampaign();

        $bouncerList = new BouncedList(
            $campaign,
            $mailingList->getEmail(),
            300,
            $message
        );

        $this
            ->getContainer()
            ->get("comitium5_mailing.repository.bounced_list")
            ->save($bouncerList);

        $this
            ->getContainer()
            ->get("event_dispatcher")
            ->dispatch(BouncedEmailEvent::POST_CREATE, new BouncedEmailEvent($bouncerList));

        $this->delete($mailingList);
    }
}
