<?php

namespace Comitium5\MailingBundle\Services\Dispatcher;

use Comitium5\MailingBundle\Event\CampaignEvent;
use Comitium5\MailingBundle\Manager\CampaignManager;
use Comitium5\MailingBundle\Model\Campaign\CampaignInterface;
use Comitium5\MailingBundle\Model\MailingList\MailingListInterface;
use Comitium5\MailingBundle\Repository\MailingListRepository;
use Comitium5\MailingBundle\Services\Builder\CampaignMessageBuilder;
use Comitium5\MailingBundle\Services\Response\CampaignResponse;
use Comitium5\MailingBundle\Services\Response\MailingListResponse;
use Comitium5\MailingBundle\Services\Sender\MailSenderInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Class CampaignDispatcher
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\MailingBundle\Services\Dispatcher
 */
class CampaignDispatcher implements  CampaignDispatcherInterface
{
    /**
     * @var MailSenderInterface
     */
    protected $sender;

    /**
     * @var EventDispatcherInterface
     */
    protected $dispatcher;

    /**
     * @var CampaignMessageBuilder
     */
    protected $builder;

    /**
     * @var int
     */
    protected $limit;

    /**
     * @var CampaignManager
     */
    protected $manager;

    /**
     * @var bool
     */
    protected $removeMailingListAfterSent;

    /**
     * @var MailingListRepository
     */
    private $mailingListRepository;

    /**
     * @var string
     */
    private $rootDir;

    /**
     * @var int
     */
    private $threads;

    /**
     * @var string
     */
    private $host;

    /**
     * CampaignDispatcher constructor.
     * @param MailSenderInterface $sender
     * @param EventDispatcherInterface $dispatcher
     * @param CampaignMessageBuilder $builder
     * @param CampaignManager $campaignManager
     * @param MailingListRepository $mailingListRepository
     */
    public function __construct(
        MailSenderInterface $sender,
        EventDispatcherInterface $dispatcher,
        CampaignMessageBuilder $builder,
        CampaignManager $campaignManager,
        MailingListRepository $mailingListRepository
    ) {
        $this->sender = $sender;
        $this->dispatcher = $dispatcher;
        $this->builder = $builder;
        $this->removeMailingListAfterSent = true;
        $this->manager = $campaignManager;
        $this->mailingListRepository = $mailingListRepository;
        $this->limit = 400;
        $this->threads = 16;
    }

    /**
     * @param $host
     * @return $this
     */
    public function setHost($host)
    {
        $this->host = $host;

        return $this;
    }

    /**
     * @param $rootDir
     * @return $this
     */
    public function setRootDir($rootDir)
    {
        $this->rootDir = $rootDir;

        return $this;
    }

    /**
     * @param $threads
     * @return $this
     */
    public function setThreads($threads)
    {
        $this->threads = $threads;

        return $this;
    }

    /**
     * @param MailSenderInterface $sender
     * @return $this
     */
    public function setSender(MailSenderInterface $sender)
    {
        $this->sender = $sender;

        return $this;
    }

    /**
     * @return MailSenderInterface
     */
    public function getSender()
    {
        return $this->sender;
    }

    /**
     * @return CampaignMessageBuilder
     */
    public function getBuilder()
    {
        return $this->builder;
    }

    /**
     * @param CampaignMessageBuilder $builder
     * @return $this
     */
    public function setBuilder($builder)
    {
        $this->builder = $builder;

        return $this;
    }

    /**
     * @param int $limit
     * @return $this
     */
    public function setLimit($limit)
    {
        $limit = intval($limit);

        if ($limit > 0) {
            $this->limit = $limit;
        }

        return $this;
    }

    /**
     * @return int
     */
    public function getLimit()
    {
        return intval($this->limit);
    }

    /**
     * @param bool $removeMailingListAfterSent
     * @return $this
     */
    public function setRemoveMailingListAfterSent($removeMailingListAfterSent)
    {
        $this->removeMailingListAfterSent = $removeMailingListAfterSent;

        return $this;
    }

    /**
     * @param CampaignInterface $campaign
     * @return CampaignResponse
     * @throws \Exception
     */
    public function dispatch(CampaignInterface $campaign)
    {
        if (!$this->rootDir) {
            throw new \Exception("Root dir must be set and should be the path where app folder is");
        }


        $failedRecipients = [];
        $result           = 0;

        $this
            ->dispatcher
            ->dispatch(
                CampaignEvent::PRE_SEND,
                new CampaignEvent($campaign, $result, $failedRecipients)
            );


        $mailingLists = $this->getMailingLists($campaign);
        $thread       = 1;
        $parameters   = [];
        $threadUsers  = ceil(
            count($mailingLists)/$this->threads
        );

        foreach ($mailingLists as $mailingList) {
            if (!$mailingList instanceof MailingListInterface) {
                throw new \Exception("Mailign list must be an instance of MailingListInterface");
            }

            //Block
            $mailingList->setBlocked(true);

            $this
                ->mailingListRepository
                ->save($mailingList);

            $parameters[] = [
                "campaignId" => $mailingList->getCampaign()->getId(),
                "email"      => $mailingList->getEmail(),
            ];

            if (
                (count($parameters)%$threadUsers) == 0
            ) {
                $this->makeCall($parameters, $thread);

                $parameters = [];
                $thread++;
            }
        }

        // Send rest mailing list
        if (count($parameters)) {
            $this->makeCall($parameters, $thread);
        }

        $this
            ->dispatcher
            ->dispatch(
                CampaignEvent::POST_SEND,
                new CampaignEvent($campaign, $result, $failedRecipients)
            );

        return new CampaignResponse($campaign, $result, $failedRecipients);
    }

    /**
     * @param array $parameters
     * @param $thread
     */
    private function makeCall(array $parameters, $thread)
    {
        $threadCmd = 'nice php -f %s/console comitium5_mailing:campaign:send-thread --thread=%s --data=%s --delete=%s --host=%s 2>&1 &';
        $data      = json_encode($parameters);

        $call = sprintf(
            $threadCmd,
            $this->rootDir,
            $thread,
            escapeshellarg($data),
            intval($this->removeMailingListAfterSent),
            $this->host
        );

        pclose(popen($call, 'r'));
    }

    /**
     * @param CampaignInterface $campaign
     * @return mixed
     */
    private function getMailingLists(CampaignInterface $campaign)
    {
        return $this
            ->mailingListRepository
            ->createQueryBuilder("mailing_lists")
            ->where("mailing_lists.campaign = :campaign")
            ->andWhere("mailing_lists.blocked = :blocked")
            ->setParameter("campaign", $campaign)
            ->setParameter("blocked", false)
            ->setFirstResult(0)
            ->setMaxResults($this->limit)
            ->getQuery()
            ->getResult();
    }

    /**
     * @param MailingListInterface $mailingList
     * @return MailingListResponse
     */
    public function dispatchMailingList(MailingListInterface $mailingList, $thread = null)
    {
        $message = $this
            ->builder
            ->build($mailingList);

        $failedRecipients = [];

        $result = $this
            ->sender
            ->send($message, $failedRecipients);

        return new MailingListResponse($mailingList, $result, $failedRecipients);
    }
}