<?php

namespace Comitium5\MailingBundle\Services\Bouncer;

use Comitium5\MailingBundle\Entity\BouncedList;
use Comitium5\MailingBundle\Event\BouncedEmailEvent;
use Comitium5\MailingBundle\Manager\CampaignManager;
use Comitium5\MailingBundle\Model\BouncedList\BouncedListInterface;
use Comitium5\MailingBundle\Model\Campaign\CampaignInterface;
use Comitium5\MailingBundle\Repository\AbstractRepository;
use Comitium5\MailingBundle\Services\Parser\BouncerParser;
use Comitium5\MailingBundle\Services\Parser\ParserInterface;
use Comitium5\MailingBundle\Services\Reader\MailReaderInterface;
use Comitium5\MailingBundle\Services\Stats\CampaignStatsComputer;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Class PersistentBouncerWorker
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\MailingBundle\Services\Bouncer
 */
class PersistentBouncerWorker implements BouncerWorkerInterface
{
    /**
     * @var MailReaderInterface
     */
    protected $reader;

    /**
     * @var AbstractRepository
     */
    protected $bouncerListRepository;

    /**
     * @var BouncerParser
     */
    protected $parser;

    /**
     * @var CampaignManager
     */
    protected $campaignManager;

    /**
     * @var CampaignStatsComputer
     */
    protected $campaignStatsComputer;

    /**
     * @var EventDispatcherInterface
     */
    protected $dispatcher;

    /**
     * PersistentBouncerWorker constructor.
     * @param MailReaderInterface $reader
     * @param AbstractRepository $repository
     * @param ParserInterface $parser
     * @param CampaignManager $campaignManager
     * @param CampaignStatsComputer $campaignStatsComputer
     * @param EventDispatcherInterface $dispatcher
     */
    public function __construct(
        MailReaderInterface $reader,
        AbstractRepository $repository,
        ParserInterface $parser,
        CampaignManager $campaignManager,
        CampaignStatsComputer $campaignStatsComputer,
        EventDispatcherInterface $dispatcher
    ) {
        $this->reader = $reader;
        $this->bouncerListRepository = $repository;
        $this->parser = $parser;
        $this->campaignManager = $campaignManager;
        $this->campaignStatsComputer = $campaignStatsComputer;
        $this->dispatcher = $dispatcher;
    }

    /**
     * @param MailReaderInterface $reader
     * @return $this
     */
    public function setMailReader(MailReaderInterface $reader)
    {
        $this->reader = $reader;

        return $this;
    }

    /**
     * @return MailReaderInterface
     */
    public function getMailReader()
    {
        return $this->reader;
    }

    /**
     * @return int
     */
    public function work()
    {
        $created = 0;

        $reader =  $this
            ->reader
            ->read();

        $allParsedData = $this
            ->parser
            ->parse($reader);

        foreach ($allParsedData as $parsedData) {
            if (!empty($parsedData["campaignId"]) && !empty($parsedData["email"])) {
                $campaign = $this
                    ->campaignManager
                    ->getRepository()
                    ->find($parsedData["campaignId"]);

                if ($campaign instanceof CampaignInterface) {
                    $bouncerList = $this
                        ->bouncerListRepository
                        ->find([
                            "campaign" => $campaign,
                            "email"    => $parsedData["email"]
                        ]);

                    if ($bouncerList instanceof  BouncedListInterface) {
                        continue;
                    }

                    $bouncerList = new BouncedList(
                        $campaign,
                        $parsedData["email"],
                        $parsedData["errorCode"],
                        $parsedData["errorMessage"]
                    );

                    $this
                        ->bouncerListRepository
                        ->save($bouncerList);

                    $this
                        ->dispatcher
                        ->dispatch(BouncedEmailEvent::POST_CREATE, new BouncedEmailEvent($bouncerList));

                    $this
                        ->campaignStatsComputer
                        ->computeBounced(
                            $campaign,
                            $parsedData["email"],
                            $parsedData["errorCode"].":".$parsedData["errorMessage"]
                        );

                    $created++;
                }
            }
        }

        return $created;
    }
}
