<?php

namespace Comitium5\MailingBundle\Entity;

use Comitium5\MailingBundle\Exception\InvalidArgumentTypeException;
use Comitium5\MailingBundle\Model\Campaign\CampaignInterface;
use Comitium5\MailingBundle\Model\Campaign\CampaignStateValue;
use Comitium5\MailingBundle\Model\Core\DateTimeTrait;
use Comitium5\MailingBundle\Model\Core\IdentifiableTrait;
use Comitium5\MailingBundle\Model\Core\PeriodDateTimeTrait;
use Comitium5\MailingBundle\Model\MailingList\MailingListInterface;
use Comitium5\MailingBundle\Services\Message;
use Comitium5\MailingBundle\Services\MessageInterface;
use Comitium5\MailingBundle\Utils\IdentifierGenerator;
use Doctrine\Common\Collections\ArrayCollection;

/**
 * Class Campaign
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\MailingBundle\Model
 */
class Campaign implements CampaignInterface
{
    use IdentifiableTrait,
        DateTimeTrait,
        PeriodDateTimeTrait;

    /**
     * @var string
     */
    protected $name;

    /**
     * @var string
     */
    protected $testAbIdentifier;

    /**
     * @var string
     */
    protected $subject;

    /**
     * @var string
     */
    protected $fromEmail;

    /**
     * @var string
     */
    protected $fromName;

    /**
     * @var string
     */
    protected $content;

    /**
     * @var string
     */
    protected $extraData;

    /**
     * @var string
     */
    protected $extraGrouping;

    /**
     * @var CampaignStateValue
     */
    protected $state;

    /**
     * @var ArrayCollection
     */
    protected $mailingLists;

    /**
     * @var MessageInterface
     */
    protected $message;

    /**
     * @var int
     */
    protected $emailsSent;

    /**
     * @var string
     */
    protected $returnPath;

    /**
     * Campaign constructor.
     * @param string $name
     * @param string $subject
     * @param mixed $from
     * @param $content
     */
    public function __construct(
        $name,
        $subject,
        $from,
        $content
    ) {
        $this->id = IdentifierGenerator::generateUniqueId();
        $this->name = $name;
        $this->subject = $subject;
        $this->fromEmail = self::resolveFromEmail($from);
        $this->fromName = self::resolveFromName($from);
        $this->content = $content;
        $this->state = new CampaignStateValue(self::STATE_DRAFT);
        $this->createdDate = new \DateTime();
        $this->mailingLists = new ArrayCollection();
    }

    /**
     * @param $instancesNumber
     * @param $name
     * @param $subject
     * @param $from
     * @param $content
     * @return array
     */
    public static function createTestAbInstances($instancesNumber, $name, $subject, $from, $content)
    {
        if (!is_numeric($instancesNumber)) {
            throw new InvalidArgumentTypeException("Instance number must be an integer");
        }

        $instances = [];
        $testAbIdentified = IdentifierGenerator::generateUniqueId();

        for ($iterator = 0; $iterator < $instancesNumber; $iterator++) {
            $instance = new self($name, $subject, $from, $content);
            $instance->testAbIdentifier = $testAbIdentified;

            $instances[] = $instance;
        }

        return $instances;
    }

    /**
     * @param $name
     * @return $this
     */
    public function setName($name)
    {
        $this->name = $name;

        return $this;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function getTestAbIdentifier()
    {
        return $this->testAbIdentifier;
    }

    /**
     * @return string
     */
    public function getSubject()
    {
        return $this->subject;
    }

    /**
     * @param mixed $subject
     * @return $this
     */
    public function setSubject($subject)
    {
        $this->subject = $subject;

        return $this;
    }

    /**
     * @return string
     */
    public function getFromName()
    {
        return $this->fromName;
    }

    /**
     * @param string $fromName
     * @return $this
     */
    public function setFromName($fromName)
    {
        $this->fromName = self::resolveFromName($fromName);

        return $this;
    }

    /**
     * @return string
     */
    public function getFromEmail()
    {
        return $this->fromEmail;
    }

    /**
     * @param string $fromEmail
     * @return $this
     */
    public function setFromEmail($fromEmail)
    {
        $this->fromEmail = self::resolveFromEmail($fromEmail);

        return $this;
    }

    /**
     * @return string
     */
    public function getContent()
    {
        return $this->content;
    }

    /**
     * @param mixed $content
     *
     * @return $this
     */
    public function setContent($content)
    {
        $this->content = $content;

        return $this;
    }

    /**
     * @return string
     */
    public function getExtraData()
    {
        return $this->extraData;
    }

    /**
     * @param string $extraData
     * @return $this
     */
    public function setExtraData($extraData)
    {
        $this->extraData = $extraData;

        return $this;
    }

    /**
     * @return string
     */
    public function getExtraGrouping()
    {
        return $this->extraGrouping;
    }

    /**
     * @param string $extraGrouping
     * @return $this
     */
    public function setExtraGrouping($extraGrouping)
    {
        $this->extraGrouping = $extraGrouping;

        return $this;
    }

    /**
     * @return string
     */
    public function getState()
    {
        return $this->state;
    }

    /**
     * @param CampaignStateValue $state
     * @return $this
     */
    public function setState(CampaignStateValue $state)
    {
        $this->state = $state;

        return $this;
    }

    /**
     * @return ArrayCollection
     */
    public function getMailingLists()
    {
        return $this->mailingLists;
    }

    /**
     * @param $email
     * @param array $replacements
     * @return null
     */
    public function addMailingList($email, array $replacements)
    {
        if ($this->getEmailFromCollection($email)->count() > 0) {
            return null;
        }

        $mailingList = new MailingList();
        $mailingList
            ->setCreatedDate(new \DateTime())
            ->setCampaign($this)
            ->setEmail($email)
            ->setReplacements($replacements);

        $this->mailingLists->add($mailingList);
    }

    /**
     * @param $email
     * @return null
     */
    public function removeMailingList($email)
    {
        $mailList = $this->getEmailFromCollection($email);

        if ($mailList->count() === 0) {
            return null;
        }

        $this->mailingLists->removeElement($mailList->first());
    }

    /**
     * @param $email
     * @return ArrayCollection
     */
    private function getEmailFromCollection($email)
    {
        return $this->mailingLists->filter(function (MailingListInterface $mailingList) use ($email) {
            if ($mailingList->getEmail() === $email) {
                return $mailingList;
            }
        });
    }

    /**
     * @return Message
     */
    public function createMessage()
    {
        $message = new Message($this->subject, $this->content, "text/html");

        $from = $this->fromEmail;

        if ($this->fromName !== null) {
            $from = [$this->fromEmail => $this->fromName];
        }

        $message
            ->setFrom($from)
            ->setReturnPath($this->returnPath ?: $this->fromEmail)
        ;

        $this->message = $message;

        return $message;
    }

    /**
     * @return MessageInterface
     */
    public function getMessage()
    {
        return $this->message;
    }

    /**
     * @param MessageInterface $message
     * @return $this
     */
    public function setMessage(MessageInterface $message)
    {
        $this->message = $message;

        return $this;
    }

    /**
     * @param $data
     * @return array
     * @throws \Exception
     */
    private static function resolveFromEmail($data)
    {
        if (is_array($data)) {
            $data = array_keys($data);

            if (!empty($data[0])) {
                $data = $data[0];
            }
        }

        if (filter_var($data, FILTER_VALIDATE_EMAIL) === false) {
            throw new \Exception("From must be a valid email");
        }

        return $data;
    }

    /**
     * @param $data
     * @return null|string
     */
    private static function resolveFromName($data)
    {
        if (is_array($data)) {
            $data = array_values($data);

            if (!empty($data[0])) {
                return $data[0];
            }
        }

        return null;
    }

    /**
     * @return int
     */
    public function getEmailsSent()
    {
        return $this->emailsSent;
    }

    /**
     * @param int $emailsSent
     * @return $this
     */
    public function setEmailsSent($emailsSent)
    {
        $this->emailsSent = $emailsSent;

        return $this;
    }

    /**
     * @return string
     */
    public function getReturnPath()
    {
        return $this->returnPath;
    }

    /**
     * @param string $returnPath
     * @return $this
     */
    public function setReturnPath($returnPath)
    {
        $this->returnPath = $returnPath;

        return $this;
    }

    /**
     * @param $abIdentifier
     * @return $this
     */
    public function setAbIdentifier($abIdentifier)
    {
        $this->testAbIdentifier = $abIdentifier;

        return $this;
    }

    /**
     * @return Campaign
     */
    public function duplicate()
    {
        $clone = clone $this;
        $clone->id = IdentifierGenerator::generateUniqueId();
        $clone->mailingLists = new ArrayCollection();

        return $clone;
    }
}
