<?php

namespace Comitium5\MailingBundle\Services\Dispatcher;

use Comitium5\MailingBundle\Event\CampaignEvent;
use Comitium5\MailingBundle\Manager\CampaignManager;
use Comitium5\MailingBundle\Model\Campaign\CampaignInterface;
use Comitium5\MailingBundle\Model\MailingList\MailingListInterface;
use Comitium5\MailingBundle\Repository\CampaignRepository;
use Comitium5\MailingBundle\Services\Builder\CampaignMessageBuilder;
use Comitium5\MailingBundle\Services\MessageInterface;
use Comitium5\MailingBundle\Services\Response\CampaignResponse;
use Comitium5\MailingBundle\Services\Response\MailingListResponse;
use Comitium5\MailingBundle\Services\Sender\MailSenderInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Class CampaignDispatcher
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\MailingBundle\Services\Dispatcher
 */
class CampaignDispatcher implements  CampaignDispatcherInterface
{
    /**
     * @var MailSenderInterface
     */
    protected $sender;

    /**
     * @var EventDispatcherInterface
     */
    protected $dispatcher;

    /**
     * @var CampaignMessageBuilder
     */
    protected $builder;

    /**
     * @var int
     */
    protected $limit;

    /**
     * @var CampaignManager
     */
    protected $manager;

    /**
     * @var bool
     */
    protected $removeMailingListAfterSent;

    /**
     * CampaignDispatcher constructor.
     * @param MailSenderInterface $sender
     * @param EventDispatcherInterface $dispatcher
     * @param CampaignMessageBuilder $builder
     * @param CampaignManager $campaignManager
     */
    public function __construct(
        MailSenderInterface $sender,
        EventDispatcherInterface $dispatcher,
        CampaignMessageBuilder $builder,
        CampaignManager $campaignManager
    ) {
        $this->sender = $sender;
        $this->dispatcher = $dispatcher;
        $this->builder = $builder;
        $this->removeMailingListAfterSent = true;
        $this->manager = $campaignManager;
    }

    /**
     * @param MailSenderInterface $sender
     * @return $this
     */
    public function setSender(MailSenderInterface $sender)
    {
        $this->sender = $sender;

        return $this;
    }

    /**
     * @return MailSenderInterface
     */
    public function getSender()
    {
        return $this->sender;
    }

    /**
     * @return CampaignMessageBuilder
     */
    public function getBuilder()
    {
        return $this->builder;
    }

    /**
     * @param CampaignMessageBuilder $builder
     * @return $this
     */
    public function setBuilder($builder)
    {
        $this->builder = $builder;

        return $this;
    }

    /**
     * @param int $limit
     * @return $this
     */
    public function setLimit($limit)
    {
        $limit = intval($limit);

        if ($limit > 0) {
            $this->limit = $limit;
        }

        return $this;
    }

    /**
     * @return int
     */
    public function getLimit()
    {
        return intval($this->limit);
    }

    /**
     * @param bool $removeMailingListAfterSent
     * @return $this
     */
    public function setRemoveMailingListAfterSent($removeMailingListAfterSent)
    {
        $this->removeMailingListAfterSent = $removeMailingListAfterSent;

        return $this;
    }

    /**
     * @param CampaignInterface $campaign
     * @return CampaignResponse
     * @throws \Exception
     */
    public function dispatch(CampaignInterface $campaign)
    {
        $failedRecipients = [];
        $result = 0;

        $this
            ->dispatcher
            ->dispatch(
                CampaignEvent::PRE_SEND,
                new CampaignEvent($campaign, $result, $failedRecipients)
            );

        $mailingLists = $campaign->getMailingLists();

        if ($this->limit > 0) {
            $mailingLists = $mailingLists->slice(0, $this->limit);
        }

        foreach ($mailingLists as $mailingList) {
            if (!$mailingList instanceof MailingListInterface) {
                throw new \Exception("Mailign list must be an instance of MailingListInterface");
            }

            try {
                $response = $this->dispatchMailingList($mailingList);

                $failedRecipients = array_merge($failedRecipients, $response->getFailedRecipients());
                $result += $response->getResult();

                // Remove mailingList from campaign after success sending
                if ($response->getResult() === 1 && $this->removeMailingListAfterSent === true) {
                    $campaign->removeMailingList($mailingList->getEmail());
                }
            } catch (\Exception $e) {
                continue;
            }

        }

        if ($this->removeMailingListAfterSent === true) {
            $this
                ->manager
                ->save($campaign);
        }

        $this
            ->dispatcher
            ->dispatch(
                CampaignEvent::POST_SEND,
                new CampaignEvent($campaign, $result, $failedRecipients)
            );

        return new CampaignResponse($campaign, $result, $failedRecipients);
    }

    /**
     * @param MailingListInterface $mailingList
     * @return MailingListResponse
     */
    public function dispatchMailingList(MailingListInterface $mailingList)
    {
        $message = $this
            ->builder
            ->build($mailingList);

        $failedRecipients = [];

        $result = $this
            ->sender
            ->send($message, $failedRecipients);

        return new MailingListResponse($mailingList, $result, $failedRecipients);
    }

    /**
     * @param MessageInterface $message
     */
    public function addMessageHeaders(MessageInterface $message)
    {

    }
}