<?php

namespace Comitium5\MailingBundle\Services\Builder;

use Comitium5\CommonWidgetsBundle\Services\Security\DataEncryption;
use Comitium5\MailingBundle\Model\MailingList\MailingListInterface;
use Comitium5\MailingBundle\Services\Message;
use Comitium5\MailingBundle\Services\Parser\CampaignContentParser;

/**
 * Class CampaignMessageBuilder
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\MailingBundle\Services\Builder
 */
class CampaignMessageBuilder
{
    const HEADER_CAMPAIGN   = "CS_CAMPAIGN_ID";
    const HEADER_EMAIL      = "CS_EMAIL";
    const TOKEN_FIELD       = "@TOKEN@";
    const CONTACT_FIELD     = "@CONTACT_ID@";
    const UNSUBSCRIBE_FIELD = "@UNSUBSCRIBE_LINK@";
    const MAIL_FIELD        = "@MAIL_URL@";

    /**
     * @var CampaignContentParser
     */
    protected $parser;

    /**
     * CampaignMessageBuilder constructor.
     * @param CampaignContentParser $parser
     */
    public function __construct(CampaignContentParser $parser)
    {
        $this->parser = $parser;
    }

    /**
     * @return CampaignContentParser
     */
    public function getParser()
    {
        return $this->parser;
    }

    /**
     * @param CampaignContentParser $parser
     * @return $this
     */
    public function setParser(CampaignContentParser $parser)
    {
        $this->parser = $parser;

        return $this;
    }

    /**
     * @param MailingListInterface $mailingList
     *
     * @return Message
     * @throws \Exception
     */
    public function build(MailingListInterface $mailingList)
    {
        $message = $mailingList
            ->getCampaign()
            ->createMessage()
        ;

        $subject = $mailingList
            ->getCampaign()
            ->getSubject()
        ;

        $body = $mailingList
            ->getCampaign()
            ->getContent()
        ;

        $this->resolveReplacementFields($mailingList);

        $subject = $this->makeReplacements($mailingList, $subject);
        $body    = $this->makeReplacements($mailingList, $body);

        //Build stats links
        $body = $this
            ->parser
            ->parse($mailingList, $body);

        $message->setSubject($subject);
        $message->setBody($body);
        $message->setTo(trim($mailingList->getEmail()));

        $headers = $message->getHeaders();
        $headers->addTextHeader(self::HEADER_CAMPAIGN, $mailingList->getCampaign()->getId());
        $headers->addTextHeader(self::HEADER_EMAIL, $mailingList->getEmail());

        return $message;
    }

    /**
     * @param MailingListInterface $mailingList
     * @param string               $content
     *
     * @return string
     */
    private function makeReplacements(MailingListInterface $mailingList, $content)
    {
        return str_replace(
            array_keys($mailingList->getReplacements()),
            array_values($mailingList->getReplacements()),
            $content
        );
    }

    /**
     * @param MailingListInterface $mailingList
     *
     * @throws \Exception
     */
    private function resolveReplacementFields(MailingListInterface $mailingList)
    {
        $replacements = $mailingList->getReplacements();

        if (!empty($replacements[self::TOKEN_FIELD]) || empty($replacements[self::CONTACT_FIELD])) {
            return;
        }

        $encryption = new DataEncryption(sprintf(
            "%s|%s",
            $replacements[self::CONTACT_FIELD],
            $mailingList->getEmail()
        ));
        $replacements[self::TOKEN_FIELD] = urlencode($encryption->encrypt());

        foreach ($replacements as $key => &$value) {
            $replacements[self::UNSUBSCRIBE_FIELD] = str_replace($key, urlencode($value), $replacements[self::UNSUBSCRIBE_FIELD]);
            $replacements[self::MAIL_FIELD] = str_replace($key, urlencode($value), $replacements[self::MAIL_FIELD]);
        }

        $mailingList->setReplacements($replacements);
    }
}
