<?php

namespace Comitium5\DesignerBundle\Helper\Widget;


use Comitium5\DesignerBundle\Model\Interfaces\Widget\WidgetInterface;

/**
 * Class WidgetHelper
 *
 * @author Carles Gómez <carles@bab-soft.com>
 * @package Comitium5\DesignerBundle\Helper\Widget
 */
class WidgetHelper
{
    const RESOURCES_FOLDER = 'Resources';
    const CSS_FOLDER = 'css';
    const JS_FOLDER = 'js';
    const IMAGE_FOLDER = 'images';
    const TRANSLATION_FOLDER = 'translations';
    const WIDGET_FOLDER = 'Widgets';
    const PUBLIC_FOLDER = 'public';
    const UPLOAD_FOLDER = 'uploads';
    const UPLOAD_STANDARD_FOLDER = 'uploadsStandards';
    const ICON_FOLDER = 'icon';
    const STANDARD_FOLDER = 'Standards';
    const WIDGET_BUNDLE_FOLDER_PATH = "src/ComitiumSuite/Bundle/CSBundle";

    /**
     * Array to target paths, the CSBundle path and each front project path
     *
     * @var array
     */
    private $bundlePath;

    private $backendBundlePath;

    /**
     * @param $bundlePath
     */
    public function __construct($bundlePath)
    {
        $realPath = realpath($bundlePath);

        $this->bundlePath        = $realPath ? : $bundlePath;
        $this->backendBundlePath = $realPath ? : $bundlePath;
    }

    /**
     * @return string
     */
    public function getBundlePath()
    {
        return $this->bundlePath;
    }

    /**
     * @param string $bundlePath
     */
    public function setBundlePath($bundlePath)
    {
        $this->bundlePath = $bundlePath;
    }

    /**
     * @return array
     */
    public function getMainBundlePath()
    {
        return $this->bundlePath;
    }

    /**
     * Path to widget folder in bundle ex: __DIR__/Widgets/Standards/Widget1
     *
     * @param WidgetInterface $widget
     *
     * @return string
     */
    public function pathInBundle(WidgetInterface $widget)
    {
        return $this->bundlePath.$this->pathToBundle($widget);
    }

    /**
     * Path to specific folder in bundle ex: __DIR__/Widgets/Standards/Widget1/Resources/public/
     *
     * @param WidgetInterface $widget
     * @param $folder
     *
     * @return string
     */
    public function pathToFolderInBundle(WidgetInterface $widget, $folder)
    {
        return $this->bundlePath.$this->pathToBundle($widget).DIRECTORY_SEPARATOR.$folder;
    }

    /**
     * Path to specific file inside a folder of bundle ex: __DIR__/Widgets/Standards/Widget1/Resources/public/css/styles.css
     *
     * @param WidgetInterface $widget
     * @param $folder
     *
     * @param $fileName
     *
     * @return string
     */
    public function pathToFileInBundle(WidgetInterface $widget, $folder, $fileName)
    {
        return $this->bundlePath.$this->pathToBundle($widget).DIRECTORY_SEPARATOR.$folder.DIRECTORY_SEPARATOR.$fileName;
    }

    /**
     * Path to public css folder in Resources ex: __DIR__/Resources/public/Widgets/Widget1/css
     *
     * @param WidgetInterface $widget
     *
     * @return string
     */
    public function resourceCssPath(WidgetInterface $widget)
    {
        return $this->pathToResourcesPublicWidgetFolder($widget).DIRECTORY_SEPARATOR.self::CSS_FOLDER;
    }

    /**
     * Path to public js folder in Resources ex: __DIR__/Resources/public/Widgets/Widget1/js
     *
     * @param WidgetInterface $widget
     *
     * @return string
     */
    public function resourceJsPath(WidgetInterface $widget)
    {
        return $this->pathToResourcesPublicWidgetFolder($widget).DIRECTORY_SEPARATOR.self::JS_FOLDER;
    }

    /**
     * Path to public images folder in Resources ex: __DIR__/Resources/public/Widgets/Widget1/images
     *
     * @param WidgetInterface $widget
     *
     * @return string
     */
    public function resourceImagesPath(WidgetInterface $widget)
    {
        return $this->pathToResourcesPublicWidgetFolder($widget).DIRECTORY_SEPARATOR.self::IMAGE_FOLDER;
    }

    /**
     * Path to public images folder in Resources ex: __DIR__/Resources/public/Widgets/Widget1/images
     *
     * @param WidgetInterface $widget
     *
     * @return string
     */
    public function resourceIconPath(WidgetInterface $widget)
    {
        return $this->pathToResourcesPublicWidgetFolder($widget).DIRECTORY_SEPARATOR.self::ICON_FOLDER;
    }

    /**
     * Path to public Resources folder ex: __DIR__/Resources
     *
     * @return string
     */
    public function pathToResourcesFolder()
    {
        return $this->bundlePath.DIRECTORY_SEPARATOR.self::RESOURCES_FOLDER;
    }

    /**
     *
     * Path to public widget folder in Resources ex: __DIR__/Resources/public/Widgets/Widget1
     *
     * @param WidgetInterface $widget
     *
     * @return string
     */
    public function pathToResourcesPublicWidgetFolder(WidgetInterface $widget)
    {
        return $this->bundlePath.
            DIRECTORY_SEPARATOR.
            self::RESOURCES_FOLDER.
            DIRECTORY_SEPARATOR.
            self::PUBLIC_FOLDER.
            DIRECTORY_SEPARATOR.
            self::WIDGET_FOLDER.
            DIRECTORY_SEPARATOR.
            $widget->formatName();
    }

    /**
     * Path to translation file ex: __DIR__/Resources/translations/Widget1.ca.yml
     *
     * @param WidgetInterface $widget
     * @param $localeCode
     *
     * @return string
     */
    public function resourceTranslationsFilePath(WidgetInterface $widget, $localeCode)
    {
        return $this->bundlePath.$this->buildRelativeResourceTranslationsPath().DIRECTORY_SEPARATOR.$widget->formatName().'.'.$localeCode.'.yml';
    }

    /**
     * Path to public translation folder in resources ex: __DIR__/Resources/translations/
     *
     * @return string
     */
    public function resourceTranslationsFolderPath()
    {
        return $this->bundlePath.$this->buildRelativeResourceTranslationsPath();
    }

    /**
     * @param string          $filePath
     * @param WidgetInterface $widget
     *
     * @return int
     */
    public function isTranslationFile(WidgetInterface $widget, $filePath)
    {
        $regex = sprintf('/^%s\/Resources\/translations\/%s\.(.+)\.yml/', $widget->formatName(), $widget->formatName());

        return (bool) preg_match($regex, $filePath);
    }

    /**
     * @param WidgetInterface $widget
     *
     * @return bool
     */
    public function isUploaded(WidgetInterface $widget)
    {
        return file_exists($this->pathInBundle($widget));
    }

    /**
     * @return string
     */
    private function buildRelativeResourceTranslationsPath()
    {
        return DIRECTORY_SEPARATOR.self::RESOURCES_FOLDER.DIRECTORY_SEPARATOR.self::TRANSLATION_FOLDER;
    }

    /**
     * @param WidgetInterface $widget
     *
     * @return string
     */
    private function pathToBundle(WidgetInterface $widget)
    {
        return DIRECTORY_SEPARATOR.self::WIDGET_FOLDER.DIRECTORY_SEPARATOR.$widget->buildFolderName(false);
    }

    /**
     *
     */
    public function resetBundlePath()
    {
        $this->bundlePath = $this->backendBundlePath;
    }
}
