<?php

namespace Comitium5\DesignerBundle\Factory\Search\Suggest;

use Comitium5\ApiClientBundle\ApiClient\ResourcesTypes;
use Comitium5\DesignerBundle\Factory\Search\Filter\DateMustFilter;
use Comitium5\DesignerBundle\Factory\Search\Filter\LocaleMustFilter;
use Comitium5\DesignerBundle\Factory\Search\Filter\PublishMustFilter;
use Comitium5\DesignerBundle\Provider\Locale\LocaleProviderInterface;
use Comitium5\DesignerBundle\ValueObjects\SearchOrderObject;
use Comitium5\DesignerBundle\ValueObjects\SearchQueryObject;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class SearchQueryObjectFactory
 *
 * @author David Carrascosa <david@bab-soft.com>
 * @package Comitium5\DesignerBundle\Factory\Search\Suggest
 */
class SearchQueryObjectFactory
{
    /**
     * @param Request $request
     * @param LocaleProviderInterface $localeProvider
     * @param array $subsites
     *
     * @return SearchQueryObject
     */
    public static function create(
        Request $request,
        LocaleProviderInterface $localeProvider,
        array $subsites
    ): SearchQueryObject
    {
        $contentType = $request->get('contentType');

        return new SearchQueryObject(
            $request->get('term'),
            self::resolveContentType($contentType),
            (int) $request->get('page', 1),
            (int) $request->get('limit', 10),
            $subsites,
            self::resolveFilters($localeProvider, $contentType),
            null,
            new SearchOrderObject()
        );
    }

    /**
     * @param string $contentType
     *
     * @return string
     */
    protected static function resolveContentType(string $contentType): string
    {
        if ($contentType === ResourcesTypes::DIRECTORY_ITEM) {
            return 'directory-items';
        }
        if ($contentType === ResourcesTypes::NEWSLETTER_TEMPLATE) {
            return 'newsletter-template';
        }

        return $contentType;
    }

    /**
     * @param LocaleProviderInterface $localeProvider
     * @param string $contentType
     *
     * @return array|string
     */
    protected static function resolveFilters(LocaleProviderInterface $localeProvider, string $contentType): array
    {
        $filters = [
            new LocaleMustFilter($localeProvider->getDefaultLocale()->getCode())
        ];

        switch($contentType) {
            case ResourcesTypes::CLASSIFIED:
                $filters[] = new PublishMustFilter(true);
                $filters[] = new DateMustFilter('endAt', new \DateTime());
                break;
            case ResourcesTypes::ACTIVITY:
                $filters[] = new PublishMustFilter(true);
                $filters[] = new DateMustFilter('unPublishDate', new \DateTime());
                break;
            case ResourcesTypes::POLL:
            case ResourcesTypes::PAGE:
            case ResourcesTypes::GALLERY:
            case ResourcesTypes::DIRECTORY_ITEM:
            case ResourcesTypes::LIVE_EVENT:
                $filters[] = new PublishMustFilter(true);
                $filters[] = new DateMustFilter('publishDate', new \DateTime(), DateMustFilter::LESS_THAN_EQUAL);
                break;
            default:
                break;
        }

        return $filters;
    }
}
