<?php

namespace Comitium5\DesignerBundle\Controller\Asset;

use Comitium5\ApiClientBundle\Client\Services\AssetCropsApiService;
use Comitium5\ApiClientBundle\Client\Services\WatermarksApiService;
use Comitium5\ApiClientBundle\ValueObject\IdentifiedValue;
use Comitium5\ApiClientBundle\ValueObject\ParametersValue;
use Comitium5\DesignerBundle\Manager\Asset\CordsManager;
use Comitium5\DesignerBundle\ValueObjects\Common\SizableObject;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Class AssetCropController
 *
 * @author Carles Gómez <carles@bab-soft.com>
 * @package Comitium5\DesignerBundle\Controller\Asset
 */
class AssetCropController extends Controller
{
    /**
     * @param Request $request
     * @param $assetId
     *
     * @return Response
     * @throws \Exception
     */
    public function indexAction(Request $request, $assetId): Response
    {
        $assetCropsApiService = new AssetCropsApiService($this->get("designer.api"));
        $assetWatermarksApiService = new WatermarksApiService($this->get("designer.api"));

        [
            "asset" => $asset,
            "crops" => $crops,
            "allCrops" => $allCrops,
        ] = $assetCropsApiService->find(new IdentifiedValue($assetId));

        $allWatermarks = $assetWatermarksApiService->findAll();

        return $this->render(
            "Comitium5DesignerBundle:AssetCrop:index.html.twig",
            [
                "asset" => $asset,
                "crops" => $crops,
                "cropsForButtons" => $this->handleAssetCropCords($asset, $allCrops),
                "allWatermarks" => $allWatermarks
            ]
        );
    }

    /**
     * @param Request $request
     * @param $assetId
     * @param $cropId
     *
     * @throws \Exception
     */
    public function createAction(Request $request, $assetId, $cropId): Response
    {
        $cords = $request->get('cords', []);
        $watermarks = $request->get('watermarks', []) ;

        $assetCropsApiService = new AssetCropsApiService($this->get("designer.api"));

        return $this->render(
            "Comitium5DesignerBundle:AssetCrop:crops-results.html.twig",
            $assetCropsApiService->generateResizes(new ParametersValue([
                "asset" => $assetId,
                "crop"  => $cropId,
                "cords" => $cords,
                "watermarks" => $watermarks,
            ]))
        );
    }

    /**
     * @param array $asset
     * @param array $allCrops
     *
     * @return array
     */
    private function handleAssetCropCords(array $asset, array $allCrops): array
    {
        $cordsManager       = new CordsManager();
        $cropsCords         = [];
        $sizableImageObject = new SizableObject(
            $asset["metadata"]["width"],
            $asset["metadata"]["height"]
        );

        foreach ($allCrops as $crop) {
            $sizableCropObject = new SizableObject($crop["width"], $crop["height"]);

            $cropsCords[] = [
                "entity" => $crop,
                "cords" => $cordsManager->generateCords(
                    $sizableImageObject,
                    $sizableCropObject
                ),
                "ratio" => $cordsManager->calculateRatio($sizableCropObject),
            ];
        }

        return $cropsCords;
    }

    /**
     * @param Request $request
     * @param $assetId
     * @param $cropId
     *
     * @return JsonResponse
     */
    public function getAppliedWatermarksAction(Request $request, $assetId, $cropId): JsonResponse
    {
        $assetWatermarksApiService = new WatermarksApiService($this->get("designer.api"));

        $assetCropWatermarks = $assetWatermarksApiService->findAssetWatermarks($assetId, $cropId);

        $watermarks["watermarks"] = \array_map(function ($watermark) {
            return $watermark['id'];
        }, $assetCropWatermarks);

        return new JsonResponse($watermarks);
    }
}
