/**
 *
 * @constructor
 */
var ModifyContentActivity = function() {
    var _this = this;

    this.selector = '[data-content-action]';
    this.url = '';
    this.module = null;
    this.event = 'click';

    this.signals = {
        _get: new signals.Signal(),
        _post: new signals.Signal(),
        _success: new signals.Signal(),
        _always: new signals.Signal(),
        _error: new signals.Signal()
    };

    this.functions = {
        get: function(params) {

            var d = Q.defer();

            _this.signals._get.dispatch(d, params);

            if(d.promise.isPending() && !params.managed) {
                $.get(params.url)
                    .success(function (data, status, jqXHR) {
                        if (data) {
                            params.response = data;

                            if (data.title) {
                                params.title = data.title;
                            }

                            if (data.message) {
                                params.message = data.message;
                            }
                        }

                        d.resolve(params);
                    })
                    .fail(function (jqXHR) {

                        var _error;

                        if(jqXHR.hasOwnProperty('responseJSON')) {
                            _error = new Error(Translator.trans(jqXHR.responseJSON.message));

                            if(jqXHR.responseJSON.hasOwnProperty('title')) {
                                _error.title = Translator.trans(jqXHR.responseJSON.title);
                            }

                        } else {
                            _error = new Error(Translator.trans(jqXHR.responseText));
                        }

                        d.reject(_error);
                    });

            }

            return d.promise;
        },
        post: function(params) {
            var d = Q.defer();
            _this.signals._post.dispatch(d, params);

            if(d.promise.isPending() || !params.managed) {
                $.post(params.url, params.postParams)
                .success(function (response) {
                    d.resolve(params);
                })
                .fail(function (jqXHR) {
                        var _error = new Error(Translator.trans(params.message));
                        _error.title = Translator.trans(params.title);
                    d.reject(_error);
                });
            }

            return d.promise;
        },
        onSuccess: function(params) {
            _this.signals._success.dispatch(params);

            if(!params.managed) {
                window.location = params.redirectUrl;
            }
        },
        onError: function (error) {

            _this.signals._error.dispatch(error);

            var title   = error.hasOwnProperty('title') ? error.title :  Translator.trans("cs_core.messages.deleting_error");
            var message = error.message;

            window.bootstrap.modal.show(title, message, Translator.trans("cs_core.action.accept"), true);

        },
        always: function(params) {
            _this.signals._always.dispatch(params);
        }
    }

};

// This is one type of inheritance
ModifyContentActivity.prototype = new Activity();

/**
 * Set selector
 * @param selector
 * @returns {DeleteContent}
 */
ModifyContentActivity.prototype.setSelector = function(selector) {
    this.selector = selector;
    return this;
};

/**
 * Set url
 * @param url
 * @returns {DeleteContent}
 */
ModifyContentActivity.prototype.setUrl = function(url) {
    this.url = url;
    return this;
};

/**
 * Set title
 * @param title
 * @returns {DeleteContent}
 */
ModifyContentActivity.prototype.setTitle = function(title) {
    this.title = title;
    return this;
};

ModifyContentActivity.prototype.setHandler = function(handlerName, handlerFunction) {

    if(typeof handlerFunction !== 'function') {
        throw new Error('handlerFunction must be a function!');
    }

    this.functions[handlerName] = handlerFunction;

    return this;
};

/**
 * Set Message
 * @param message
 * @returns {DeleteContent}
 */
ModifyContentActivity.prototype.setMessage = function(message) {
    this.message = message;
    return this;
};

/**
 *
 * @param redirectUrl
 * @returns {DeleteContent}
 */
ModifyContentActivity.prototype.setRedirectUrl = function(redirectUrl) {
    this.redirectUrl = redirectUrl;
    return this;
};

ModifyContentActivity.prototype.setEvent = function(event) {
    this.event = event;
    return this;
};

/**
 * @param params
 */
ModifyContentActivity.prototype.call = function(params) {
    Q.fcall(this.functions.get.bind(this), params)
        .then(this.functions.post.bind(this))
        .then(this.functions.onSuccess.bind(this))
        .catch(this.functions.onError.bind(this))
        .done(this.functions.always.bind(this));
};

/**
 * Initializes DOM
 * @returns {DeleteContent}
 */
ModifyContentActivity.prototype.init = function() {

    var _this = this;

    $(this.selector).on(this.event, function (e) {
        e.preventDefault();

        var el = $(this);

        var params = {
            element: el,
            module: el.data('module') ? el.data('module') : _this.module,
            url: el.data('url') ? el.data('url') : _this.url,
            title: el.data('title') ? el.data('title') : _this.title,
            message: el.data('message') ? el.data('message') : _this.message,
            redirectUrl: el.data('redirect-url') ? el.data('redirect-url') : _this.redirectUrl,
            managed: false,
            initialEvent: e
        };

        _this.call(params);
    });

    return this;
};

ModifyContentActivity.prototype.addParamToUrl = function(params, paramToAdd) {

    var pattern = new RegExp(/\?/g);
    var res = pattern.test(params.url);
    var signal = res ? '&' : '?';

    params.url += signal + paramToAdd;

    return this;
};
