<?php

namespace Comitium5\DesignerBundle\DataFixtures\ORM\Template;

use Comitium5\DesignerBundle\DataFixtures\Abstracts\AbstractFixture;
use Comitium5\DesignerBundle\DataFixtures\ORM\Breakpoint\BreakpointFixtures;
use Comitium5\DesignerBundle\Entity\Layout;
use Comitium5\DesignerBundle\Model\Interfaces\Template\TemplateInterface;
use Comitium5\DesignerBundle\Model\TemplateTypes;
use Doctrine\Common\DataFixtures\DependentFixtureInterface;
use Doctrine\Common\Persistence\ObjectManager;

/**
 * Class TemplateFixtures.
 *
 * @author Oscar Jimenez <oscarjg19.developer@gmail.com>
 */
class TemplateFixtures extends AbstractFixture implements DependentFixtureInterface
{
    const REFERENCE_REGEX = "/\[reference:(.+)\]/";

    /**
     * Load data fixtures with the passed EntityManager.
     *
     * @param ObjectManager $manager
     */
    public function load(ObjectManager $manager)
    {
        $factory = $this
            ->container
            ->get('designer.factory.template');

        foreach ($this->templates() as $input) {
            /**
             * @var TemplateInterface $template
             */
            $template = $factory->create();

            $template->setName($input['name']);
            $template->setDescription($input['description']);
            $template->setType($input['type']);
            $template->setWidth($input['width']);
            $template->setGutterWidth($input['gutterWidth']);
            $template->setColor($input['color']);
            $template->setColumns($input['columns']);
            $template->setHtml($input['html']);
            $template->setEnabled($input['enabled']);
            $template->setPosition($input['position']);
            $template->setUseBootstrapRows($input['useBootstrapRows']);

            foreach ($input["breakpoints"] as $breakpoint) {
                $template->addBreakpoint($this->getReference($breakpoint));
            }

            foreach ($input["layouts"] as $layout) {
                $template->addLayout($this->getReference($layout));
            }

            if ($input['parent'] !== null) {
                $template->setParent($this->getReference($input['parent']));
            }

            if ($input['group'] !== null) {
                $template->setGroup($this->getReference($input['group']));
            }

            $this->setReference($input['reference'], $template);

            $manager->persist($template);
        }

        $manager->flush();
    }

    /**
     * @return array
     */
    private function templates(): array
    {
        return [
            [
                "name" => "Base",
                "description" => "Template base",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("base.html.twig"),
                "position" => 1,
                "useBootstrapRows" => true,
                "breakpoints" => [
                    "breakpoint-xs",
                    "breakpoint-sm",
                    "breakpoint-md",
                    "breakpoint-lg",
                ],
                "parent" => null,
                "group" => null,
                "reference" => 'template-base',
                "layouts" => [],
            ],
            [
                "name" => "Home",
                "description" => "Template home",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("home.html.twig"),
                "position" => 1,
                "useBootstrapRows" => true,
                "breakpoints" => [],
                "parent" => "template-base",
                "group" => null,
                "reference" => 'template-home',
                "layouts" => [
                    "layout-header",
                    "layout-footer",
                    "layout-home",
                    "layout-newsletter",
                ],
            ],
            [
                "name" => "Thematic",
                "description" => "Template thematic",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("thematic.html.twig"),
                "position" => 2,
                "useBootstrapRows" => true,
                "breakpoints" => [],
                "parent" => "template-base",
                "group" => null,
                "reference" => 'template-thematic',
                "layouts" => [
                    "layout-header",
                    "layout-footer",
                    "layout-breadcrumb",
                    "layout-thematic-left",
                    "layout-thematic-right",
                    "layout-newsletter",
                ],
            ],
            [
                "name" => "Article",
                "description" => "Template article",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("article.html.twig"),
                "position" => 3,
                "useBootstrapRows" => true,
                "breakpoints" => [],
                "parent" => "template-base",
                "group" => null,
                "reference" => 'template-article',
                "layouts" => [
                    "layout-header",
                    "layout-footer",
                    "layout-breadcrumb",
                    "layout-article-top",
                    "layout-article-left",
                    "layout-article-right",
                    "layout-newsletter",
                ],
            ],
            [
                "name" => "Live event",
                "description" => "Template live event",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("live-event.html.twig"),
                "position" => 4,
                "useBootstrapRows" => true,
                "breakpoints" => [],
                "parent" => "template-base",
                "group" => null,
                "reference" => 'template-live-event',
                "layouts" => [
                    "layout-header",
                    "layout-footer",
                    "layout-breadcrumb",
                    "layout-live-top",
                    "layout-live-left",
                    "layout-live-right",
                    "layout-newsletter",
                ],
            ],
            [
                "name" => "Static",
                "description" => "Template static",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("static.html.twig"),
                "position" => 5,
                "useBootstrapRows" => true,
                "breakpoints" => [],
                "parent" => "template-base",
                "group" => null,
                "reference" => 'template-static',
                "layouts" => [
                    "layout-header",
                    "layout-footer",
                    "layout-static",
                ],
            ],
            [
                "name" => "AMP",
                "description" => "Template amp",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1024,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("amp.html.twig"),
                "position" => 2,
                "useBootstrapRows" => false,
                "breakpoints" => [
                    "breakpoint-xs",
                    "breakpoint-sm",
                    "breakpoint-md",
                    "breakpoint-lg",
                ],
                "parent" => null,
                "group" => null,
                "reference" => 'template-amp',
                "layouts" => [
                    "layout-amp",
                ],
            ],
            [
                "name" => "AMP News list",
                "description" => "Template amp news list",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1024,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("amp-news-list.html.twig"),
                "position" => 3,
                "useBootstrapRows" => false,
                "breakpoints" => [
                    "breakpoint-xs",
                    "breakpoint-sm",
                    "breakpoint-md",
                    "breakpoint-lg",
                ],
                "parent" => null,
                "group" => null,
                "reference" => 'template-amp-news-list',
                "layouts" => [
                    "layout-amp-news-list",
                ],
            ],
            [
                "name" => "Newsletter",
                "description" => "Template newsletter",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1024,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("newsletter.html.twig"),
                "position" => 4,
                "useBootstrapRows" => false,
                "breakpoints" => [
                    "breakpoint-xs",
                    "breakpoint-sm",
                    "breakpoint-md",
                    "breakpoint-lg",
                ],
                "parent" => null,
                "group" => null,
                "reference" => 'template-newsletter',
                "layouts" => [
                    "layout-newsletter",
                ],
            ],
            [
                "name" => "Search",
                "description" => "Template search",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("search.html.twig"),
                "position" => 5,
                "useBootstrapRows" => true,
                "breakpoints" => [],
                "parent" => "template-base",
                "group" => null,
                "reference" => 'template-search',
                "layouts" => [
                    "layout-header",
                    "layout-footer",
                    "layout-search",
                ],
            ],
            [
                "name" => "Error",
                "description" => "Template error",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("error.html.twig"),
                "position" => 5,
                "useBootstrapRows" => true,
                "breakpoints" => [],
                "parent" => "template-base",
                "group" => null,
                "reference" => 'template-error',
                "layouts" => [
                    "layout-header",
                    "layout-footer",
                    "layout-error",
                ],
            ],
            [
                "name" => "Gallery",
                "description" => "Template gallery",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("gallery.html.twig"),
                "position" => 5,
                "useBootstrapRows" => true,
                "breakpoints" => [],
                "parent" => "template-base",
                "group" => null,
                "reference" => 'template-gallery',
                "layouts" => [
                    "layout-header",
                    "layout-footer",
                    "layout-breadcrumb",
                    "layout-gallery",
                    "layout-newsletter",
                ],
            ],
            [
                "name" => "Register login account",
                "description" => "Register login account",
                "type" => TemplateTypes::TEMPLATE,
                "width" => 1300,
                "gutterWidth" => 30,
                "color" => "#00aceb",
                "columns" => 12,
                "enabled" => true,
                "html" => $this->readTemplateFile("register-login-account.html.twig"),
                "position" => 5,
                "useBootstrapRows" => true,
                "breakpoints" => [],
                "parent" => "template-base",
                "group" => null,
                "reference" => 'template-register-login-account',
                "layouts" => [
                    "layout-header",
                    "layout-footer",
                    "layout-account",
                ],
            ],
        ];
    }

    /**
     * @return array
     */
    public function getDependencies()
    {
        return [
            TemplateGroupFixtures::class,
            BreakpointFixtures::class,
        ];
    }

    /**
     * @param string $filename
     * @return string
     */
    private function readTemplateFile(string $filename): string
    {
        $filePath = __DIR__.DIRECTORY_SEPARATOR."Resources".DIRECTORY_SEPARATOR.$filename;

        $template = \file_get_contents($filePath);
        if (false === $template){

            return '';
        }

        return $this->handleLayoutReferences($template);
    }

    /**
     * @param string $template
     *
     * @return string
     */
    private function handleLayoutReferences(string $template): string
    {
        \preg_match_all(self::REFERENCE_REGEX, $template, $matches);

        if ([] !== $matches[1]){
            foreach ($matches[1] as $key => $reference) {
                /** @var Layout $layout */
                $layout = $this->getReference($reference);
                $id = $layout->getId();
                if (null !== $id) {
                    $template = \str_replace($matches[0][$key], $id, $template);
                }
            }
        }

        return $template;
    }
}
