<?php

namespace Comitium5\DesignerBundle\Helper;

/**
 * Class IcoUtils
 *
 * @author Carles Gómez <carles@bab-soft.com>
 * @package Comitium5\DesignerBundle\Helper
 */
class IcoUtils
{
    const ICO_DEFAULT_PREFIX     = "cs-ico cs-ico-";
    const ICO_ENTITY_TYPE_PREFIX = "cs-ico cs-ico-t";
    const ICO_TRANS_KEY_PREFIX   = "cs_content_type.types.";

    /**
     * @param $entity
     * @return string
     */
    public function resolveIcoByEntity($entity)
    {
        if ($entity instanceof AssetInterface) {
            $ico = $this->resolveAssetIcon($entity);
        } elseif ($entity instanceof SelfContentTypeInterface) {
            $ico = $this
                ->icoByContentType($entity->getSelfContentType());
        } else {
            $ico = $this
                ->icoByNamespace(get_class($entity));
        }

        return $ico;
    }

    /**
     * Resolve preferred ico by entity namespace
     *
     * @param $namespaceOrName
     * @return string
     */
    public function ico($namespaceOrName)
    {
        if (!empty($namespaceOrName)) {
            $constant = strtoupper(Utils::getEntityNameType($namespaceOrName));

            if (empty($constant)) {
                $constant = $namespaceOrName;
            }

            $constantValue =  Utils::resolveEntityTypeCode($namespaceOrName);

            if ($constantValue) {
                return self::ICO_ENTITY_TYPE_PREFIX . $this->parseClass($constantValue);
            }

            $constantValue =  Utils::resolveEntityType($namespaceOrName);

            if ($constantValue) {
                return self::ICO_DEFAULT_PREFIX . $this->parseClass($constantValue);
            }

            return self::ICO_DEFAULT_PREFIX . $this->parseClass($constant);
        }
    }

    /**
     * @param $namespace
     * @return string
     */
    public function icoByNamespace($namespace)
    {
        if (!empty($namespace)) {
            $entityName = strtolower(Utils::getEntityNameType($namespace));

            return self::ICO_DEFAULT_PREFIX . $this->parseClass($entityName);
        }
    }

    /**
     * @param $contentType
     * @return string
     */
    public function icoByContentType($contentType)
    {
        if (!empty($contentType)) {
            return self::ICO_ENTITY_TYPE_PREFIX . $this->parseClass($contentType);
        }
    }

    /**
     * @param $prefix
     * @return string
     */
    public function icoCustom($prefix)
    {
        return self::ICO_DEFAULT_PREFIX . $this->parseClass($prefix);
    }

    /**
     * @param ModuleInterface $module
     * @return string
     */
    public function icoFromModule(ModuleInterface $module)
    {
        return $module->getContentType() ?
            $this->icoByContentType($module->getContentType()):
            $this->icoCustom($module->getName());
    }

    /**
     * @param $name
     * @return string
     */
    public function icoTranslatorKey($name)
    {
        return self::ICO_TRANS_KEY_PREFIX . $name;
    }

    /**
     * @param $name
     * @return mixed|string
     */
    private function parseClass($name)
    {
        $name = trim($name);
        $name = strtolower($name);
        $name = str_replace("_", "-", $name);

        return $name;
    }

    /**
     * @param AssetInterface $asset
     * @return string
     */
    private function resolveAssetIcon(AssetInterface $asset)
    {
        $type = $asset->getType();

        switch ($type) {
            case AssetTypes::VIDEO:
                $ico = $this
                    ->icoByContentType(ContentType::ASSET_VIDEOS);
                break;
            case AssetTypes::DOCUMENT:
                $ico = $this
                    ->icoByContentType(ContentType::ASSET_DOCUMENTS);
                break;
            case AssetTypes::SOUND:
                $ico = $this
                    ->icoByContentType(ContentType::ASSET_SOUNDS);
                break;
            default:
                $ico = $this
                    ->icoByContentType(ContentType::ASSET_IMAGES);
                break;
        }

        return $ico;
    }
}
