<?php

namespace Comitium5\CommonWidgetsBundle\Tests\PayWall\Security;

use Comitium5\CommonWidgetsBundle\PayWall\Security\PayWallVoter;
use Comitium5\CommonWidgetsBundle\Services\Security\ComitiumUser;
use Comitium5\CommonWidgetsBundle\Tests\PayWall\PayWallTestCase;

/**
 * Class PayWallVoterTest
 *
 * @author Óscar Jimenez <oscar@bab-soft.com>
 * @package Comitium5\CommonWidgetsBundle\Tests\PayWall\Security
 */
class PayWallVoterTest extends PayWallTestCase
{
    /**
     * @test
     */
    public function hardPayWall()
    {
        $voter = new PayWallVoter();

        $hasAccess = $voter->hasAccess(
            $this->payWallObject(),
            $this->loggedWithAccessUser()
        );

        $this->assertTrue($hasAccess);

        $voter = new PayWallVoter();

        $hasAccess = $voter->hasAccess(
            $this->payWallObject(),
            $this->loggedWithAccessUserAndSomeExpired()
        );

        $this->assertTrue($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject(),
            $this->loggedWithNotAccessUser()
        );

        $this->assertFalse($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject(),
            $this->loggedWithAccessUserButExpired()
        );

        $this->assertFalse($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject(),
            $this->loggedWithExpiredAccessUser()
        );

        $this->assertFalse($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject(),
            $this->notLoggedUser()
        );

        $this->assertFalse($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject(),
            $this->loggedWithoutSubcriptions()
        );

        $this->assertFalse($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject(),
            $this->loggedWithAccessUserAndExpirationDateOnNull()
        );

        $this->assertTrue($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject(),
            $this->notLoggedUser(),
            'google'
        );

        $this->assertTrue($hasAccess);
    }

    /**
     * @test
     */
    public function counterPayWall()
    {
        $voter = new PayWallVoter();

        $payWallOptions = [
            'type' => 'counter',
            'data' => [
                'counterLimit' => 3
            ]
        ];

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->loggedWithAccessUser()
        );

        $this->assertTrue($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->loggedWithAccessUserAndSomeExpired()
        );

        $this->assertTrue($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->loggedWithNotAccessUser([
                'comitium-paywall-counter' => implode(',', [
                    'http://foo.com',
                    'http://foo.com',
                    'http://foo.com',
                ])
            ])
        );

        $this->assertTrue($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->loggedWithNotAccessUser([
                'comitium-paywall-counter' => implode(',', [
                    'http://foo.com',
                    'http://foo.com',
                    'http://foo.com',
                    'http://foo.com',
                ])
            ])
        );

        $this->assertFalse($hasAccess);
    }

    public function testShouldReturnTrueWhenDateLimitIsLessThanNow()
    {
        $voter = new PayWallVoter();

        $payWallOptions = [
            'type' => 'date',
            'data' => [
                'dateLimit' => (new \DateTime('now -1 minutes'))->format("d/m/Y H:i"),
            ],
        ];

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->notLoggedUser()
        );

        $this->assertTrue($hasAccess);
    }

    public function testShouldReturnFalseWhenDateLimitIsLaterThanNow()
    {
        $voter = new PayWallVoter();

        $payWallOptions = [
            'type' => 'date',
            'data' => [
                'dateLimit' => (new \DateTime('now +1 minutes'))->format("d/m/Y H:i"),
            ],
        ];

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->notLoggedUser()
        );

        $this->assertFalse($hasAccess);
    }

    public function testShouldReturnTrueWhenFormatDateIsValid()
    {
        $voter = new PayWallVoter();

        $payWallOptions = [
            'type' => 'date',
            'data' => [
                'dateLimit' => (new \DateTime('now -1 minutes'))->format("Y-m-d H:i"),
                'dateFormat' => "Y-m-d H:i",
            ],
        ];

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->notLoggedUser()
        );

        $this->assertTrue($hasAccess);
    }

    public function testShouldReturnFalseWhenFormatDateDoesNotMatch()
    {
        $voter = new PayWallVoter();

        $payWallOptions = [
            'type' => 'date',
            'data' => [
                'dateLimit' => (new \DateTime('now -1 minutes'))->format("Y-m-d H:i"),
                'dateFormat' => "d/m/Y H:i",
            ],
        ];

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->notLoggedUser()
        );

        $this->assertFalse($hasAccess);
    }

    public function testShouldReturnFalseWhenFormatDateIsNotValid()
    {
        $voter = new PayWallVoter();

        $payWallOptions = [
            'type' => 'date',
            'data' => [
                'dateLimit' => (new \DateTime('now -1 minutes'))->format("Y-m-d H:i"),
                'dateFormat' => 12352123,
            ],
        ];

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->notLoggedUser()
        );

        $this->assertFalse($hasAccess);
    }

    public function testShouldReturnTrueWhenUserIsLoggedAndHasSubscriptionAndDateIsLessThanNow()
    {
        $voter = new PayWallVoter();

        $payWallOptions = [
            'type' => 'date',
            'data' => [
                'dateLimit' => (new \DateTime('now -1 minutes'))->format("d/m/Y H:i"),
            ],
        ];

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->loggedWithAccessUser()
        );

        $this->assertTrue($hasAccess);

        $hasAccess = $voter->hasAccess(
            $this->payWallObject($payWallOptions),
            $this->loggedWithAccessUserAndSomeExpired()
        );

        $this->assertTrue($hasAccess);

    }


    /**
     * @param array $cookies
     *
     * @return ComitiumUser
     * @throws \Exception
     */
    private function loggedWithAccessUser(array $cookies = [])
    {
        $user = new ComitiumUser([
            'id' => 1,
            'subscriptions' => [
                [
                    "id" => 100,
                    "expirationDate" => (new \DateTime('now +2 days'))->format("Y-m-d H:i:s"),
                ],
            ],
        ]);

        $user->setCookies($cookies);

        return $user;
    }

    /**
     * @param array $cookies
     *
     * @return ComitiumUser
     * @throws \Exception
     */
    private function loggedWithAccessUserAndSomeExpired(array $cookies = [])
    {
        $user = new ComitiumUser([
            'id' => 1,
            'subscriptions' => [
                [
                    "id" => 100,
                    "expirationDate" => (new \DateTime('now -1 days'))->format("Y-m-d H:i:s"),
                ],
                [
                    "id" => 100,
                    "expirationDate" => (new \DateTime('now -2 days'))->format("Y-m-d H:i:s"),
                ],
                [
                    "id" => 100,
                    "expirationDate" => (new \DateTime('now +2 days'))->format("Y-m-d H:i:s"),
                ],
                [
                    "id" => 200,
                    "expirationDate" => (new \DateTime('now -10 days'))->format("Y-m-d H:i:s"),
                ],
            ],
        ]);

        $user->setCookies($cookies);

        return $user;
    }

    /**
     * @param array $cookies
     *
     * @return ComitiumUser
     * @throws \Exception
     */
    private function loggedWithNotAccessUser(array $cookies = [])
    {
        $user = new ComitiumUser([
            'id' => 1,
            'subscriptions' => [
                [
                    "id" => 800,
                    "expirationDate" => (new \DateTime('now +2 days'))->format("Y-m-d H:i:s"),
                ],
                [
                    "id" => 600,
                    "expirationDate" => (new \DateTime('now +4 days'))->format("Y-m-d H:i:s"),
                ],
            ],
        ]);

        $user->setCookies($cookies);

        return $user;
    }

    /**
     * @param array $cookies
     *
     * @return ComitiumUser
     * @throws \Exception
     */
    private function loggedWithAccessUserButExpired(array $cookies = [])
    {
        $user = new ComitiumUser([
            'id' => 1,
            'subscriptions' => [
                [
                    "id" => 800,
                    "expirationDate" => (new \DateTime('now +2 days'))->format("Y-m-d H:i:s"),
                ],
                [
                    "id" => 200,
                    "expirationDate" => (new \DateTime('now -2 days'))->format("Y-m-d H:i:s"),
                ],
                [
                    "id" => 100,
                    "expirationDate" => (new \DateTime('now -4 days'))->format("Y-m-d H:i:s"),
                ],
            ],
        ]);

        $user->setCookies($cookies);

        return $user;
    }

    /**
     * @param array $cookies
     *
     * @return ComitiumUser
     */
    private function loggedWithAccessUserAndExpirationDateOnNull(array $cookies = []): ComitiumUser
    {
        $user = new ComitiumUser([
            'id' => 1,
            'subscriptions' => [
                [
                    "id" => 100,
                    "expirationDate" => null,
                ],
            ],
        ]);

        $user->setCookies($cookies);

        return $user;
    }

    /**
     * @param array $cookies
     *
     * @return ComitiumUser
     * @throws \Exception
     */
    private function loggedWithExpiredAccessUser(array $cookies = [])
    {
        $user = new ComitiumUser([
            'id' => 1,
            'subscriptions' => [
                [
                    "id" => 200,
                    "expirationDate" => (new \DateTime('now -2 days'))->format("Y-m-d H:i:s"),
                ],
                [
                    "id" => 100,
                    "expirationDate" => (new \DateTime('now -4 days'))->format("Y-m-d H:i:s"),
                ],
            ],
        ]);

        $user->setCookies($cookies);

        return $user;
    }

    /**
     * @return array
     */
    private function notLoggedUser()
    {
        return new ComitiumUser();
    }

    /**
     * @param array $cookies
     *
     * @return ComitiumUser
     */
    private function loggedWithoutSubcriptions(array $cookies = [])
    {
        $user = new ComitiumUser([
            'id' => 1,
            'subscriptions' => []
        ]);

        $user->setCookies($cookies);

        return $user;
    }

}
