<?php

namespace Comitium5\CommonWidgetsBundle\PayWall\Worker;

use Comitium5\CommonWidgetsBundle\Cache\MemoryCacheInterface;
use Comitium5\CommonWidgetsBundle\PayWall\Token\PayWallToken;
use Comitium5\CommonWidgetsBundle\PayWall\ValueObject\PayWallObject;
use Symfony\Component\Templating\EngineInterface;

/**
 * Class PayWallWorker
 *
 * @author Óscar Jimenez <oscar@bab-soft.com>
 * @package Comitium5\CommonWidgetsBundle\PayWall\Worker
 */
class PayWallWorker
{
    /**
     * @var EngineInterface
     */
    protected $templating;

    /**
     * @var MemoryCacheInterface
     */
    protected $cache;

    /**
     * @var PayWallToken
     */
    private $payWallToken;

    /**
     * PayWallWorker constructor.
     *
     * @param EngineInterface $templating
     * @param MemoryCacheInterface $cache
     * @param PayWallToken $payWallToken
     */
    public function __construct(
        EngineInterface $templating,
        MemoryCacheInterface $cache,
        PayWallToken $payWallToken
    ) {
        $this->templating   = $templating;
        $this->cache        = $cache;
        $this->payWallToken = $payWallToken;
    }

    /**
     * @param PayWallObject $payWallObject
     *
     * @return string
     * @throws \ReflectionException
     */
    public function workAndGetToken(PayWallObject $payWallObject)
    {
        $this->renderPrivateContentAndSaveInCache($payWallObject);

        $this->renderPayWallContentAndSaveInCache($payWallObject);

        return $this
            ->payWallToken
            ->encrypt($payWallObject);
    }

    /**
     * @param PayWallObject $payWallObject
     *
     * @return mixed|string|null
     */
    public function fetchPrivateContent(PayWallObject $payWallObject)
    {
        $cacheKey = $this->generatePrivateContentKey($payWallObject);

        $content = $this
            ->cache
            ->get($cacheKey);

        if (empty($content)) {
            $content = $this->renderPrivateContentAndSaveInCache($payWallObject);
        }

        return $content;
    }

    /**
     * @param PayWallObject $payWallObject
     *
     * @return mixed|string|null
     * @throws \Exception
     */
    public function fetchPayWallContent(PayWallObject $payWallObject)
    {
        $cacheKey = $this->generatePayWallContentKey($payWallObject);

        $content = $this
            ->cache
            ->get($cacheKey);

        if (empty($content)) {
            $content = $this->renderPayWallContentAndSaveInCache($payWallObject);
        }

        return $content;
    }

    /**
     * @param PayWallObject $payWallObject
     *
     * @return string
     */
    private function renderPrivateContentAndSaveInCache(PayWallObject $payWallObject)
    {
        $content = $this->generateContent(
            $payWallObject->getPrivateContentTemplate(),
            $payWallObject->getPrivateContentData()
        );

        $this->saveInCache(
            $this->generatePrivateContentKey($payWallObject),
            $content,
            $this->cacheTTL($payWallObject)
        );

        return $content;
    }

    /**
     * @param PayWallObject $payWallObject
     *
     * @return string
     */
    private function renderPayWallContentAndSaveInCache(PayWallObject $payWallObject)
    {
        $content = $this->generateContent(
            $payWallObject->getPayWallTemplate(),
            $payWallObject->getPayWallTemplateData()
        );

        $this->saveInCache(
            $this->generatePayWallContentKey($payWallObject),
            $content,
            $this->cacheTTL($payWallObject)
        );

        return $content;
    }

    /**
     * @param $template
     * @param array $data
     *
     * @return string
     */
    private function generateContent($template, array $data)
    {
        return $this
            ->templating
            ->render($template, $data);
    }

    /**
     * @param $key
     * @param $content
     * @param $ttl
     *
     * @return bool
     */
    private function saveInCache($key, $content, $ttl)
    {
        return $this
            ->cache
            ->set($key, $content, $ttl);
    }

    /**
     * @param PayWallObject $payWallObject
     *
     * @return string
     */
    private function generatePrivateContentKey(PayWallObject $payWallObject): string
    {
        $data = $payWallObject->getPayWallTemplateData();

        return sprintf(
            '%s%s%s_%s',
            $payWallObject->getCacheKeyPrefix(),
            $payWallObject->getObjectType(),
            !empty($data['locale']) ? '_' . $data['locale'] : '',
            $payWallObject->getObjectId()
        );
    }

    /**
     * @param PayWallObject $payWallObject
     *
     * @return string
     */
    private function generatePayWallContentKey(PayWallObject $payWallObject): string
    {
        $payWallData = $payWallObject->getPayWallTemplateData();

        return sprintf(
            '%spaywall-template-%s__%s',
            $payWallObject->getCacheKeyPrefix(),
            md5($payWallObject->getPayWallTemplate()),
            md5($payWallData ? json_encode($payWallObject->getPayWallTemplateData()) : '')
        );
    }

    /**
     * @param PayWallObject $payWallObject
     *
     * @return int
     */
    private function cacheTTL(PayWallObject $payWallObject)
    {
        $ttl       = (int) $payWallObject->getCacheTTL();
        $increment = (int) $payWallObject->getCacheTTLSecondsToIncrease();

        if ($ttl > 0) {
            return $ttl + $increment;
        }

        return 300;
    }
}