<?php

namespace Tests\Services;

use Comitium5\ApiClientBundle\Client\Client;
use Comitium5\CommonWidgetsBundle\Tests\TestCase;
use Comitium5\CommonWidgetsBundle\Services\GATemplateDataResolver;

/**
 * Class GATemplateDataResolverTest
 *
 * @author Óscar Jimenez <oscar@bab-soft.com>
 * @package Tests\Services
 */
class GATemplateDataResolverTest extends TestCase
{
    /**
     * @param $entity
     * @param $expected
     *
     * @throws \Exception
     *
     * @test
     * @dataProvider dataV1
     */
    public function resolveV1($entity, $expected)
    {
        $gaData = (new GATemplateDataResolver($entity, 1, 1))->resolve();

        $this->assertEquals($gaData, $expected);
    }

    /**
     * @param $entity
     * @param $expected
     *
     * @throws \Exception
     *
     * @test
     * @dataProvider dataV2
     */
    public function resolveV2($entity, $expected)
    {
        $gaData = (new GATemplateDataResolver($entity, 1, 2, $this->apiClientStub()))->resolve();

        $this->assertEquals($gaData, $expected);
    }

    /**
     * @param $entity
     * @param $expected
     *
     * @throws \Exception
     *
     * @test
     * @dataProvider emptyData
     */
    public function resolveWithEmptyData($entity, $expected)
    {
        $gaData = (new GATemplateDataResolver($entity, 1, 2, $this->apiClientStub()))->resolve();

        $this->assertEquals($gaData, $expected);

        $gaData = (new GATemplateDataResolver($entity, 1, 1))->resolve();

        $this->assertEquals($gaData, $expected);
    }

    /**
     * @param $entity
     * @param $expected
     *
     * @throws \Exception
     *
     * @test
     * @dataProvider dataV2
     */
    public function shouldThrowException($entity, $expected)
    {
        $this->expectExceptionMessage('Api client must be set if api version is 2');

        new GATemplateDataResolver($entity, 1, 2);
    }

    /**
     * @return array
     */
    public function dataV1()
    {
        return [
            [
                [
                    'id' => 1,
                    'tags' => [],
                    'categories' => [],
                    'author' => null,
                    'userData' => [
                        'id' => 1,
                        'acronym' => 'F.B'
                    ],
                    'selfContentType' => 102,
                    'type' => 1,
                ],
                [
                    'dimension1'  => '',
                    'dimension2'  => '',
                    'dimension3'  => '102',
                    'dimension4'  => '1',
                    'dimension6'  => '1',
                    'dimension7'  => '',
                    'dimension8'  => '',
                    'dimension9'  => '',
                    'dimension10' => '',
                    'dimension11' => '1',
                    'dimension12' => 'F.B',
                    'dimension13' => '1',
                ]
            ],
            [
                [
                    'id' => 1,
                    'tags' => [
                        [
                            'id' => 1,
                            'lang' => [
                                'title' => 'Tag 1'
                            ]
                        ],
                        [
                            'id' => 2,
                            'lang' => [
                                'title' => 'Tag 2'
                            ]
                        ],
                    ],
                    'categories' => [
                        [
                            'id' => 11,
                            'lang' => [
                                'title' => 'Cat 11'
                            ]
                        ],
                        [
                            'id' => 22,
                            'lang' => [
                                'title' => 'Cat 22'
                            ]
                        ],
                    ],
                    'author' => [
                        'id' => 33,
                        'name' => 'Foo',
                        'suername' => '',
                    ],
                    'userData' => [
                        'id' => 1,
                        'acronym' => 'F.B'
                    ],
                    'selfContentType' => 102,
                    'type' => 1,
                ],
                [
                    'dimension1'  => '1',
                    'dimension2'  => '11',
                    'dimension3'  => '102',
                    'dimension4'  => '1',
                    'dimension6'  => '1',
                    'dimension7'  => 'Tag 1',
                    'dimension8'  => 'Cat 11',
                    'dimension9'  => '33',
                    'dimension10' => 'Foo',
                    'dimension11' => '1',
                    'dimension12' => 'F.B',
                    'dimension13' => '1',
                ]
            ],
            [
                [
                    'id' => 1,
                    'tags' => [
                        [
                            'id' => 1,
                            'lang' => [
                                'title' => 'Tag 1'
                            ]
                        ],
                        [
                            'id' => 2,
                            'lang' => [
                                'title' => 'Tag 2'
                            ]
                        ],
                    ],
                    'categories' => [
                        [
                            'id' => 11,
                            'lang' => [
                                'title' => 'Cat 11'
                            ]
                        ],
                        [
                            'id' => 22,
                            'lang' => [
                                'title' => 'Cat 22'
                            ]
                        ],
                    ],
                    'author' => [
                        'id' => 33,
                        'name' => 'Foo',
                        'surname' => 'Bar',
                    ],
                    'userData' => [
                        'id' => 1,
                        'acronym' => 'F.B'
                    ],
                    'selfContentType' => 102,
                    'type' => 1,
                ],
                [
                    'dimension1'  => '1',
                    'dimension2'  => '11',
                    'dimension3'  => '102',
                    'dimension4'  => '1',
                    'dimension6'  => '1',
                    'dimension7'  => 'Tag 1',
                    'dimension8'  => 'Cat 11',
                    'dimension9'  => '33',
                    'dimension10' => 'Foo Bar',
                    'dimension11' => '1',
                    'dimension12' => 'F.B',
                    'dimension13' => '1',
                ]
            ],
            [
                [
                    'id' => 1,
                    'tags' => [
                        [
                            'id' => 1,
                            'lang' => [
                                'title' => 'Tag 1'
                            ]
                        ],
                        [
                            'id' => 2,
                            'lang' => [
                                'title' => 'Tag 2'
                            ]
                        ],
                    ],
                    'categories' => [
                        [
                            'id' => 11,
                            'lang' => [
                                'title' => 'Cat 11'
                            ]
                        ],
                        [
                            'id' => 22,
                            'lang' => [
                                'title' => 'Cat 22'
                            ]
                        ],
                    ],
                    'author' => [
                        'id' => 33,
                        'name' => 'Foo',
                        'surname' => 'Bar',
                    ],
                    'user' => 1,
                    'selfContentType' => 102,
                    'type' => 1,
                ],
                [
                    'dimension1'  => '1',
                    'dimension2'  => '11',
                    'dimension3'  => '102',
                    'dimension4'  => '1',
                    'dimension6'  => '1',
                    'dimension7'  => 'Tag 1',
                    'dimension8'  => 'Cat 11',
                    'dimension9'  => '33',
                    'dimension10' => 'Foo Bar',
                    'dimension11' => '',
                    'dimension12' => '',
                    'dimension13' => '1',
                ]
            ],
        ];
    }

    /**
     * @return array
     */
    public function dataV2()
    {
        return [
            [
                [
                    'id' => 1,
                    'tags' => [],
                    'categories' => [],
                    'author' => null,
                    'user' => [
                        'id' => 1,
                        'acronym' => 'F.B'
                    ],
                    'selfContentType' => 102,
                    'type' => [
                        'id' => 1
                    ],
                ],
                [
                    'dimension1'  => '',
                    'dimension2'  => '',
                    'dimension3'  => '102',
                    'dimension4'  => '1',
                    'dimension6'  => '1',
                    'dimension7'  => '',
                    'dimension8'  => '',
                    'dimension9'  => '',
                    'dimension10' => '',
                    'dimension11' => '1',
                    'dimension12' => 'F.B',
                    'dimension13' => '1',
                ]
            ],
            [
                [
                    'id' => 1,
                    'tags' => [
                        [
                            'id' => 1,
                        ],
                        [
                            'id' => 2,
                        ],
                    ],
                    'categories' => [
                        [
                            'id' => 11,
                        ],
                        [
                            'id' => 22,
                        ],
                    ],
                    'author' => [
                        'id' => 33,
                    ],
                    'user' => [
                        'id' => 1,
                        'acronym' => 'F.B'
                    ],
                    'selfContentType' => 102,
                    'type' => [
                        'id' => 1
                    ],
                ],
                [
                    'dimension1'  => '1',
                    'dimension2'  => '11',
                    'dimension3'  => '102',
                    'dimension4'  => '1',
                    'dimension6'  => '1',
                    'dimension7'  => 'Tag Foo',
                    'dimension8'  => 'Category Foo',
                    'dimension9'  => '33',
                    'dimension10' => 'Author Foo',
                    'dimension11' => '1',
                    'dimension12' => 'F.B',
                    'dimension13' => '1',
                ]
            ],
        ];
    }

    /**
     * @return array
     */
    public function emptyData()
    {
        return [
            [
                [
                    'id' => 1,
                ],
                [
                    'dimension1'  => '',
                    'dimension2'  => '',
                    'dimension3'  => '',
                    'dimension4'  => '',
                    'dimension6'  => '1',
                    'dimension7'  => '',
                    'dimension8'  => '',
                    'dimension9'  => '',
                    'dimension10' => '',
                    'dimension11' => '',
                    'dimension12' => '',
                    'dimension13' => '1',
                ]
            ],
        ];
    }

    /**
     * @return \PHPUnit\Framework\MockObject\MockObject
     * @throws \ReflectionException
     */
    private function apiClientStub()
    {
        $stub = $this->createMock(
            Client::class
        );

        $stub
            ->expects($this->any())
            ->method('find')
            ->willReturnCallback(function ($type, $id){
                $response = [];

                switch ($type) {
                    case 'tags':
                        $response = [
                            'title' => "Tag Foo",
                            'id'    => $id,
                        ];
                        break;
                    case 'categories':
                        $response = [
                            'title' => "Category Foo",
                            'id'    => $id,
                        ];
                        break;
                    case 'authors':
                        $response = [
                            'fullName' => "Author Foo",
                            'id'       => $id,
                        ];
                        break;
               }

               return [
                   'statusCode' => 200,
                   'data'       => $response,
               ];
            });

        return $stub;
    }
}