<?php

namespace Comitium5\CommonWidgetsBundle\PayWall\ValueObject;

/**
 * Class PayWallTokenObject
 *
 * @author Óscar Jimenez <oscar@bab-soft.com>
 * @package Comitium5\CommonWidgetsBundle\PayWall\ValueObject
 */
class PayWallObject
{
    const PAYWALL_HARD_TYPE    = 'hard';

    const PAYWALL_COUNTER_TYPE = 'counter';

    const GOOGLE_SOURCE = 'google';

    /**
     * @var integer
     */
    private $objectId;

    /**
     * @var string
     */
    private $objectType;

    /**
     * @var array
     */
    private $objectSubscriptions;

    /**
     * @var string
     */
    private $privateContentTemplate;

    /**
     * @var array
     */
    private $privateContentData;

    /**
     * @var string
     */
    private $payWallTemplate;

    /**
     * @var array
     */
    private $payWallTemplateData;

    /**
     * @var string
     */
    private $cacheKeyPrefix;

    /**
     * @var null|int
     */
    private $cacheTTL;

    /**
     * @var int
     */
    private $cacheTTLSecondsToIncrease;

    /**
     * @var array
     */
    private $payWallOptions;

    /**
     * PayWallObject constructor.
     *
     * @param $objectId
     * @param $objectType
     * @param $objectSubscriptions
     * @param $privateContentTemplate
     * @param $privateContentData
     * @param $payWallTemplate
     * @param $payWallTemplateData
     * @param $cacheTTL
     * @param int $cacheTTLSecondsToIncrease
     * @param array $payWallOptions
     * @param string $cacheKeyPrefix
     *
     * @throws \Exception
     */
    public function __construct(
        $objectId,
        $objectType,
        $objectSubscriptions,
        $privateContentTemplate,
        $privateContentData,
        $payWallTemplate,
        $payWallTemplateData,
        $cacheTTL,
        $cacheTTLSecondsToIncrease = 120,
        array $payWallOptions = [],
        $cacheKeyPrefix = '__PAYWALL__'
    ) {
        $this->objectId = $objectId;
        $this->objectType = $objectType;
        $this->objectSubscriptions = $objectSubscriptions;
        $this->privateContentTemplate = $privateContentTemplate;
        $this->privateContentData = $privateContentData;
        $this->payWallTemplate = $payWallTemplate;
        $this->payWallTemplateData = $payWallTemplateData;
        $this->cacheTTL = $cacheTTL;
        $this->cacheTTLSecondsToIncrease = $cacheTTLSecondsToIncrease;
        $this->cacheKeyPrefix = $cacheKeyPrefix;

        $this->applyDefaultPayWallOptions($payWallOptions);
    }

    /**
     * @return false|string
     * @throws \ReflectionException
     */
    public function serialize()
    {
        $reflection = new \ReflectionClass($this);

        $props = $reflection->getProperties(\ReflectionProperty::IS_PRIVATE);

        $data = [];

        foreach ($props as $prop) {
            $data[$prop->name] = $this->{$prop->name};
        }

        return json_encode($data);
    }

    /**
     * @return int
     */
    public function getObjectId()
    {
        return $this->objectId;
    }

    /**
     * @return string
     */
    public function getObjectType()
    {
        return $this->objectType;
    }

    /**
     * @return array
     */
    public function getObjectSubscriptions()
    {
        return $this->objectSubscriptions;
    }

    /**
     * @return string
     */
    public function getPrivateContentTemplate()
    {
        return $this->privateContentTemplate;
    }

    /**
     * @return array
     */
    public function getPrivateContentData()
    {
        return $this->privateContentData;
    }

    /**
     * @return string
     */
    public function getPayWallTemplate()
    {
        return $this->payWallTemplate;
    }

    /**
     * @return array
     */
    public function getPayWallTemplateData()
    {
        return $this->payWallTemplateData;
    }

    /**
     * @return string
     */
    public function getCacheKeyPrefix()
    {
        return $this->cacheKeyPrefix;
    }

    /**
     * @return int|null
     */
    public function getCacheTTL()
    {
        return $this->cacheTTL;
    }

    /**
     * @return int
     */
    public function getCacheTTLSecondsToIncrease()
    {
        return $this->cacheTTLSecondsToIncrease;
    }

    /**
     * @return array
     */
    public function getPayWallOptions()
    {
        return $this->payWallOptions;
    }

    /**
     * @param array $payWallOptions
     *
     * @return array
     * @throws \Exception
     */
    private function applyDefaultPayWallOptions(array $payWallOptions)
    {
        if (empty($payWallOptions)) {
            $payWallOptions = [
                'type' => self::PAYWALL_HARD_TYPE,
                'data' => [],
            ];
        }

        if (!isset($payWallOptions['type']) || !isset($payWallOptions['data'])) {
            throw new \Exception('Type and data must be provided for paywall options');
        }

        $this->validateOptionsType($payWallOptions['type']);
        $this->validateOptionsData($payWallOptions['type'], $payWallOptions['data']);

        $this->payWallOptions = $payWallOptions;
    }

    /**
     * @param $type
     *
     * @throws \Exception
     */
    private function validateOptionsType($type)
    {
        $validTypes = [
            self::PAYWALL_COUNTER_TYPE,
            self::PAYWALL_HARD_TYPE,
        ];

        $isValid = in_array($type, $validTypes);

        if (!$isValid) {
            throw new \Exception(
                sprintf('Invalid type %s. Type must be one of: %s',
                    $type,
                    implode(',', $validTypes)
                )
            );
        }
    }

    /**
     * @param $type
     * @param $data
     *
     * @throws \Exception
     */
    private function validateOptionsData($type, $data)
    {
        if (!is_array($data)) {
            throw new \Exception('Data option type must be an array');
        }

        if ($type === self::PAYWALL_COUNTER_TYPE) {
            if (empty($data['counterLimit'])) {
                throw new \Exception(
                    sprintf('counterLimit option must be provided for counter type. Options provided under data are: %s',
                        implode(',', array_keys($data))
                    )
                );
            }
        }
    }
}
