<?php

namespace Comitium5\CommonWidgetsBundle\Controller;

use Comitium5\CommonWidgetsBundle\PayWall\Security\PayWallVoter;
use Comitium5\CommonWidgetsBundle\PayWall\Token\PayWallToken;
use Comitium5\CommonWidgetsBundle\PayWall\Worker\PayWallWorker;
use Comitium5\CommonWidgetsBundle\Services\Security\ComitiumUser;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorage;

/**
 * Class PayWallController
 *
 * @author Óscar Jimenez <oscar@bab-soft.com>
 * @package Comitium5\CommonWidgetsBundle\Controller
 */
class PayWallController
{
    /**
     * @var TokenStorage
     */
    protected $tokenStorage;

    /**
     * @var PayWallToken
     */
    protected $payWallToken;

    /**
     * @var PayWallWorker
     */
    protected $payWallWorker;

    /**
     * @var PayWallVoter
     */
    private $payWallVoter;

    /**
     * @var
     */
    protected $startTime;


    /**
     * PayWallController constructor.
     *
     * @param TokenStorage $tokenStorage
     * @param PayWallToken $payWallToken
     * @param PayWallWorker $payWallWorker
     * @param PayWallVoter $payWallVoter
     */
    public function __construct(
        TokenStorage $tokenStorage,
        PayWallToken $payWallToken,
        PayWallWorker $payWallWorker,
        PayWallVoter $payWallVoter
    ){
        $this->tokenStorage  = $tokenStorage;
        $this->payWallToken  = $payWallToken;
        $this->payWallWorker = $payWallWorker;
        $this->payWallVoter  = $payWallVoter;
    }

    /**
     * @return Response
     */
    public function checkAction()
    {
        $this->startTime = microtime(true);

        try {
            return $this->handleResponse();
        } catch (\Throwable $e) {
            return JsonResponse::create([
                'allowed' => false,
                'content' => null,
            ], Response::HTTP_BAD_REQUEST);
        }
    }

    /**
     * @return Response
     * @throws \Exception
     */
    private function handleResponse()
    {
        $payWallObject = $this->getPayWallObjectFromToken();

        $hasAccess = $this
            ->payWallVoter
            ->hasAccess(
                $payWallObject,
                $this->getUser()
            );

        if ($hasAccess === true) {
            $content = $this
                ->payWallWorker
                ->fetchPrivateContent($payWallObject);

            return $this->buildResponse($content, $hasAccess);
        }

        $content = $this
            ->payWallWorker
            ->fetchPayWallContent($payWallObject);

        return $this->buildResponse($content, $hasAccess);
    }

    /**
     * @return ComitiumUser
     */
    private function getUser()
    {
        $token = $this
            ->tokenStorage
            ->getToken();

        $user = $token->getUser();

        if ($user instanceof ComitiumUser) {
            $cookies = $this
                ->getRequest()
                ->cookies
                ->all();

            $user->setCookies($cookies);

            return $user;
        }

        return new ComitiumUser();
    }

    /**
     * @return \Comitium5\CommonWidgetsBundle\PayWall\ValueObject\PayWallObject
     * @throws \Exception
     */
    private function getPayWallObjectFromToken()
    {
        $token = $this
            ->getRequest()
            ->request
            ->get('token', null);

        return $this
            ->payWallToken
            ->decrypt($token);
    }

    /**
     * @param $content
     * @param $allowed
     *
     * @return JsonResponse
     */
    private function buildResponse($content, $allowed)
    {
        $response = new JsonResponse([
            'content' => $content,
            'allowed' => $allowed,
            'time'    => (microtime(true) - $this->startTime),
        ]);

        return $response;
    }

    /**
     * @return Request
     */
    private function getRequest()
    {
        return Request::createFromGlobals();
    }
}
