<?php

namespace Comitium5\CommonWidgetsBundle\Tests\Utils;

use Comitium5\CommonWidgetsBundle\Tests\TestCase;
use Comitium5\CommonWidgetsBundle\Utils\Assets\AssetsUtils;
use Comitium5\CommonWidgetsBundle\Utils\Common\CommonUtils;

/**
 * Class CommonUtilsTest
 *
 * phpunit Tests/Utils/CommonUtilsTest.php
 *
 * @author Oscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\ApiClientBundle\Tests\ApiClient
 */
class CommonUtilsTest extends TestCase
{
    public function testGetFieldsByKey()
    {
        $fieldValues = [
            [
                "field" => [
                    "key" => "foo_bar_key",
                ],
                "value" => "foo",
            ],
            [
                "field" => [
                    "key" => "another_test",
                ],
                "value" => "foo",
            ],
            [
                "field" => [
                    "key" => "other_key",
                ],
                "value" => "foo",
            ],
        ];

        $matches = CommonUtils::getFieldsByKey($fieldValues, "another_test");

        $this->assertEquals(1, count($matches));
    }

    public function testGetFieldsByType()
    {
        $fieldValues = [
            [
                "field" => [
                    "key" => "foo_bar_key",
                ],
                "value" => "foo",
            ],
            [
                "field" => [
                    "key" => "foo_test",
                ],
                "value" => "foo",
            ],
            [
                "field" => [
                    "key" => "other_key",
                ],
                "value" => "foo",
            ],
        ];

        $matches = CommonUtils::getFieldsByType($fieldValues, "foo");

        $this->assertEquals(2, count($matches));
    }

    public function testGetAssetResize()
    {
        $asset = [
            "id" => 1,
            "children" => [],
        ];

        $resize = CommonUtils::getAssetResize($asset, [200, 200]);

        $this->assertEquals($resize["id"], 1);

        $asset = [
            "id" => 1,
            "children" => [
                [
                    "id" => 2,
                        "metadata" => [
                            "width" => 200,
                            "height" => 200,
                        ],
                ],
            ],
        ];

        $resize = CommonUtils::getAssetResize($asset, [200, 200]);

        $this->assertEquals($resize["id"], 2);

        $resize = CommonUtils::getAssetResize($asset, 100);

        $this->assertEquals($resize["id"], 1);
    }

    /**
     * phpunit --filter testGenerateProSrcSet Tests/Utils/CommonUtilsTest.php
     */
    public function testGenerateProSrcSet()
    {
        $asset = [
            "children" => [
                ["metadata" => ["width" => 1600, "height" => 400], "url" => "http://foo.bar"],
                ["metadata" => ["width" => 800, "height"  => 200], "url" => "http://foo.bar"],
                ["metadata" => ["width" => 400, "height"  => 100], "url" => "http://foo.bar"],
                ["metadata" => ["width" => 400, "height"  => 100], "url" => "http://foo.bar"],
            ],
        ];

        $matches = ['1000|250,800|200'];

        $src = AssetsUtils::generateProSrcSet($asset, $matches);

        $this->assertEquals($src, "800|200://foo.bar");

        $matches = ['400|100,800|200'];

        $src = AssetsUtils::generateProSrcSet($asset, $matches);

        $this->assertEquals($src, "800|200://foo.bar,400|100://foo.bar");

        $matches = [];

        $src = AssetsUtils::generateProSrcSet($asset, $matches);

        $this->assertEquals($src, "1600|400://foo.bar,800|200://foo.bar,400|100://foo.bar");

        $matches = ['50|50,20|20'];

        $src = AssetsUtils::generateProSrcSet($asset, $matches);

        $this->assertEquals($src, "");
    }

    /**
     * phpunit --filter testTruncateWords Tests/Utils/CommonUtilsTest.php
     */
    public function testTruncateWords()
    {
        $input = "a b c d e f g h i j k";

        $output = CommonUtils::truncateWords($input, 5);

        $this->assertEquals("a b c d e...", $output);

        $output = CommonUtils::truncateWords($input, 20);

        $this->assertEquals("a b c d e f g h i j k", $output);
    }

    /**
     * phpunit --filter testTruncateHtmlEntitiesWords Tests/Utils/CommonUtilsTest.php
     */
    public function testTruncateHtmlEntitiesWords()
    {
        $input = 'foo &nbsp;&eacute;s <strong>bar</strong> d&#39;una';

        $output = CommonUtils::cleanText($input);

        $this->assertEquals("foo és bar d'una", $output);
    }

    /**
     * phpunit --filter testBuildFieldKeyToElastic Tests/Utils/CommonUtilsTest.php
     */
    public function testBuildFieldKeyToElastic()
    {
        $input = [
            "field" => [
                "key" => "foo"
            ]
        ];

        $output = CommonUtils::buildFieldKeyToElastic($input);

        $this->assertEquals("*_foo", $output);
    }
}
