<?php

namespace Comitium5\CommonWidgetsBundle\Twig;

use Comitium5\CommonWidgetsBundle\Services\GoogleApiService;
use Comitium5\CommonWidgetsBundle\Utils\Common\CommonUtils;
use Comitium5\CommonWidgetsBundle\Utils\RichSnippets\ActivityRichSnippets;
use Comitium5\CommonWidgetsBundle\Utils\RichSnippets\ArticleRichSnippets;
use Knp\Component\Pager\Pagination\PaginationInterface;
use Knp\Component\Pager\Paginator;

/**
 * Class CommonExtension
 *
 * @author Oscar Jimenez <oscarjg19.developer@gmail.com>
 * @package Comitium5\CommonWidgetsBundle\Twig
 */
class CommonExtension extends \Twig_Extension implements \Twig_Extension_GlobalsInterface
{
    const PAGINATE_LIMIT = 10;

    /**
     * @var PaginationInterface
     */
    private $paginator;

    /**
     * @var string
     */
    private $isProdEnvironment;

    /**
     * CommonExtension constructor
     *
     * @param Paginator $paginator
     * @param $isProdEnvironment
     * @param GoogleApiService $googleApiService
     */
    public function __construct(
        Paginator $paginator,
        $isProdEnvironment,
        GoogleApiService $googleApiService
    )
    {
        $this->paginator = $paginator;
        $this->isProdEnvironment = $isProdEnvironment;
        $this->googleApiService = $googleApiService;
    }

    /**
     * @return array
     */
    public function getGlobals() {

        return [
            'is_prod_environment' => $this->isProdEnvironment
        ];
    }

    /**
     * @return array
     */
    public function getFilters()
    {
        return [
            new \Twig_SimpleFilter('truncate_words', [$this, 'truncateWords']),
            new \Twig_SimpleFilter('html_to_text', [$this, 'htmlToText']),
        ];
    }

    /**
     * @return array
     */
    public function getFunctions()
    {
        return [
            new \Twig_SimpleFunction('cs_pagination_ajax', [$this, 'paginationAjax'], ['is_safe' => ['html'], 'needs_environment' => true]),
            new \Twig_SimpleFunction('cs_field', [$this, 'fields'], ['is_safe' => ['html'], 'needs_environment' => true]),
            new \Twig_SimpleFunction('cs_get_fields_by_typology', [$this, 'getFieldType']),
            new \Twig_SimpleFunction('cs_get_related_content_by_type', [$this, 'getRelatedContentByType']),
            new \Twig_SimpleFunction('cs_get_fields_by_key', [$this, 'getFieldByKey']),
            new \Twig_SimpleFunction('cs_asset_resize', [$this, 'getAssetResize']),
            new \Twig_SimpleFunction('cs_get_children', [$this, 'getChildren']),
            new \Twig_SimpleFunction('cs_get_loops', [$this, 'getLoops']),
            new \Twig_SimpleFunction('cs_article_google_rich_snippet', array($this, 'articleGoogleRichSnippet')),
            new \Twig_SimpleFunction('cs_activity_google_rich_snippet', array($this, 'activityGoogleRichSnippet')),
            new \Twig_SimpleFunction('json_decode', array($this, 'jsonDecode')),
            new \Twig_SimpleFunction('datetime_from_array', array($this, 'instanceDateTimeFromArray')),
            new \Twig_SimpleFunction('google_shorter_url', array($this, 'googleShorterUrl')),
            new \Twig_SimpleFunction('truncate_words', [$this, 'truncateWords']),
            new \Twig_SimpleFunction('html_to_text', [$this, 'htmlToText']),
        ];
    }

    /**
     * @param \Twig_Environment $environment
     * @param array $apiResponse
     * @param $domSelectorElement
     * @param $controller
     * @param $action
     * @param array $parameters
     * @param null $renderTemplate
     * @param null $paginationTemplate
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function paginationAjax(
        \Twig_Environment $environment,
        array $apiResponse,
        $domSelectorElement,
        $controller,
        $action,
        array $parameters,
        $renderTemplate = null,
        $paginationTemplate = null
    ) {
        $pagination = $this->paginator->paginate(
            $apiResponse['results'],
            $apiResponse['page'],
            empty($apiResponse['limit']) ? self::PAGINATE_LIMIT : $apiResponse['limit']
        );

        $pagination->setTotalItemCount($apiResponse['total']);

        return $environment->render(
            $renderTemplate ?: 'Comitium5CommonWidgetsBundle:Paginate:pagination_ajax.html.twig', [
                'pagination' => $pagination,
                'domSelector' => $domSelectorElement,
                'controller' => $controller,
                'action' => $action,
                'parameters' => \json_encode($parameters),
                'page' => $apiResponse['page'],
                'pages' => $apiResponse['pages'],
                'template' => $paginationTemplate ?: 'Comitium5CommonWidgetsBundle:Paginate:theme/bootstrap_v3_pagination.html.twig',
            ]
        );
    }

    /**
     * @param \Twig_Environment $environment
     * @param $type
     * @param $name
     * @param $content
     * @param string $id
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function fields(\Twig_Environment $environment, $type, $name, $content, $id = '')
    {
        return $environment->render(
            'Comitium5CommonWidgetsBundle:Fields:visual_editor_field.html.twig', [
                'type' => $type,
                'name' => $name,
                'content' => $content,
                'id' => $id
            ]
        );
    }

    /**
     * @param array $input
     * @param $typology
     *
     * @return array
     */
    public function getFieldType(array $input, $typology)
    {
        return CommonUtils::getFieldsByType($input, $typology);
    }

    /**
     * @param array $input
     * @param $key
     *
     * @return array
     */
    public function getFieldByKey(array $input, $key)
    {
        return CommonUtils::getFieldsByKey($input, $key);
    }

    /**
     * @param array $input
     * @param $className
     *
     * @return array
     */
    public function getRelatedContentByType(array $input, $className)
    {
        return CommonUtils::getRelatedContentByType($input, $className);
    }

    /**
     * @param array $asset
     * @param $sizes
     * @param bool|true $strict
     * @return array
     */
    public function getAssetResize(array $asset, $sizes, $strict = true)
    {
        return CommonUtils::getAssetResize($asset, $sizes, $strict);
    }

    /**
     * @param array         $asset
     * @param integer|array $sizes
     *
     * @return array
     */
    public function getChildren(array $input)
    {
        return CommonUtils::getChildren($input);
    }

    /**
     * @param array $entity
     * @return array
     */
    public function getLoops(array $entity)
    {
        return CommonUtils::getLoops($entity);
    }

    /**
     * @param array $dateTime
     * @return \DateTime
     */
    public function instanceDateTimeFromArray(array $dateTime)
    {
        return CommonUtils::instanceDateTimeFromArray($dateTime);
    }

    /**
     * @param array $article
     * @param array $defaultValues
     *
     * @return array
     */
    public function articleGoogleRichSnippet(array $article, array $defaultValues = [])
    {
        return ArticleRichSnippets::googleRichSnippet($article, $defaultValues);
    }

    /**
     * @param array $activity
     * @return array
     */
    public function activityGoogleRichSnippet(array $activity)
    {
        return ActivityRichSnippets::googleRichSnippet($activity);
    }

    /**
     * @param $str
     * @param bool $array
     * @return mixed
     */
    public function jsonDecode($str, $array = true)
    {
        return json_decode($str, $array);
    }

    /**
     * @param $string
     * @param $numberOfWord
     * @param string $separator
     * @param int $firstWord
     * @return mixed|string
     */
    public function truncateWords($string, $numberOfWord, $separator = '...', $firstWord = 0)
    {
        return CommonUtils::truncateWords($string, $numberOfWord, $separator, $firstWord);
    }

    /**
     * @param $string
     * @return string
     */
    public function htmlToText($string)
    {
        return CommonUtils::cleanText($string);
    }

    /**
     * @param $url
     * @return string
     */
    public function googleShorterUrl($url)
    {
        return $this->googleApiService->generateShortedUrl($url);
    }

    /**
     * Returns the name of the extension.
     *
     * @return string The extension name
     */
    public function getName()
    {
        return "comitium5_common_widgets_bundle_pagination_twig_extension";
    }
}
