<?php

namespace Comitium5\CommonWidgetsBundle\Cache;

use Predis\Client as RedisClient;
use Predis\Response\Status;

/**
 * Class CSRedisWrapper
 *
 * Manage the redis session
 *
 * @link http://redis.io/
 * @link https://github.com/nrk/predis
 * @link https://github.com/snc/SncRedisBundle
 *
 * @author Oscar Jimenez <oscarjg19.developer@gmail.com>
 * @package Comitium5\CommonWidgetsBundle\Cache
 */
class RedisWrapper implements MemoryCacheInterface
{
    /**
     * The code when a request is correct
     *
     * @name RESPONSE_OK
     */
    const RESPONSE_OK = 'OK';

    /**
     * @var Client $client The redis client
     * @access private
     */
    private $client;

    /**
     * @var string $host the host of the redis client
     * @access public
     */
    public $host;

    /**
     * @var string $scheme the type of connection
     * @access public
     */
    public $scheme;

    /**
     * @var string $port the port for the redis connection
     * @access public
     */
    public $port;

    /**
     * @var int
     * @access public
     */
    public $database;

    /**
     * @param $host
     * @param $scheme
     * @param $port
     * @param $dataBase
     */
    public function __construct($host, $scheme, $port, $dataBase)
    {
        $this->host = $host;
        $this->scheme = $scheme;
        $this->port = $port;
        $this->database = $dataBase;

        $this->client = new RedisClient(
            [
                "host" => $host,
                "scheme" => $scheme,
                "port" => $port,
                "database" => $dataBase,
            ]
        );
    }

    /**
     * Set a key to the redis database
     *
     * @access public
     *
     * @param string $key   name of the redis key
     * @param string $value value to save
     * @param int    $ttl   (optional) expiration
     * @param string $index (optional) if is a collection
     *
     * @return bool
     */
    public function set($key, $value, $ttl = null, $index = null)
    {
        $success = false;

        if ($this->isConnected()) {
            $value = serialize($value);

            if ($index) {
                $response = $this->client->hset($key, $index, $value);
            } else {
                $response = $this->client->set($key, $value);
            }

            $success = $this->checkResponse($response);
            if ($success) {
                if ($ttl) {
                    $success = $this->setTTL($key, $ttl);
                }
            }
        }

        return $success;
    }

    /**
     * Get a key from the redis database
     *
     * @access public
     *
     * @param string $key   the key name
     * @param string $index (optional) if the key pertain to a index
     *
     * @return mixed|null
     */
    public function get($key, $index = null)
    {
        $response = null;

        if ($this->isConnected()) {
            if ($index) {
                $response = $this->client->hget($key, $index);
            } else {
                $response = $this->client->get($key);
            }
        }

        $response = unserialize($response);

        return $response;
    }

    /**
     * @param $key
     * @return null
     */
    public function keys($key)
    {
        $response = null;

        if ($this->isConnected()) {
            $response = $this->client->keys($key);
        }

        return $response;
    }

    /**
     * Delete a key from redis
     *
     * @access public
     *
     * @param string $key   the key name
     * @param string $index (optional) if the key pertain to a index
     *
     * @return bool
     */
    public function delete($key, $index = null)
    {
        $success = false;

        if ($this->isConnected()) {
            if ($index) {
                $result = $this->client->hdel($key, array($index));
            } else {
                $result = $this->client->del(array($key));
            }

            $success = ($result > 0) ? true : false;
        }

        return $success;
    }

    /**
     * Set a expiration to the key
     *
     * @access public
     *
     * @param string $key the key name
     * @param int    $ttl the time in seconds to expire
     *
     * @return bool
     */
    public function setTTL($key, $ttl)
    {
        $success = false;

        if ($this->isConnected()) {
            $success = $this->client->expire($key, $ttl);
        }

        return $success;
    }

    /**
     * Check if the given key exists
     *
     * @access public
     *
     * @param string $key   the key name
     * @param string $index (optional) if the key pertain to a index
     *
     * @return int|mixed
     */
    public function exists($key, $index = null)
    {
        $success = false;

        if ($this->isConnected()) {
            if ($index) {
                $success = $this->client->hexists($key, $index);
            } else {
                $success = $this->client->exists($key);
            }
        }

        return $success;
    }

    /**
     * Check if the response is ok/ko
     *
     * @access public
     *
     * @param mixed $response the response of the operation
     *
     * @return bool
     */
    protected function checkResponse($response)
    {
        $success = $response && ($response instanceof Status && self::RESPONSE_OK === $response->getPayload());

        return $success;
    }

    /**
     * Check if the client is connected, and if not, connect
     *
     * @access public
     *
     * @return bool
     */
    private function isConnected()
    {
        $success = $this->client->isConnected();
        if (!$success) {
            $this->client->connect();
            $success = $this->client->isConnected();
        }

        return $success;
    }
}
