<?php

namespace Comitium5\CommonWidgetsBundle\Services;

use Comitium5\ApiClientBundle\ApiClient\ResourcesTypes;
use Comitium5\ApiClientBundle\Client\Client;

/**
 * Class GATemplateDataResolver
 *
 * @author Óscar Jimenez <oscar@bab-soft.com>
 * @package Services
 */
class GATemplateDataResolver
{
    /**
     * @var array
     */
    protected $entity;

    /**
     * @var int
     */
    protected $apiVersion;

    /**
     * @var integer
     */
    protected $subSiteId;

    /**
     * @var Client|null
     */
    protected $apiV2Client;

    /**
     * GATemplateDataResolver constructor.
     *
     * @param array $entity
     * @param $subSiteId
     * @param $apiVersion
     * @param Client|null $apiV2Client
     *
     * @throws \Exception
     */
    public function __construct(
        array $entity,
        $subSiteId,
        $apiVersion,
        Client $apiV2Client = null
    ) {
        $this->entity      = $entity;
        $this->subSiteId   = $subSiteId;
        $this->apiVersion  = $apiVersion;
        $this->apiV2Client = $apiV2Client;

        $this->validate();
    }

    /**
     * @return array
     * @throws \Exception
     */
    public function resolve()
    {
        return array_map(function($value){
            return $this->normalizeData($value);
        }, $this->buildResponse());
    }

    /**
     * @throws \Exception
     */
    protected function validate()
    {
        if ($this->apiVersion === 2 && $this->apiV2Client === null) {
            throw new \Exception('Api client must be set if api version is 2');
        }
    }

    /**
     * @return array
     * @throws \Exception
     */
    protected function buildResponse()
    {
        return [
            'dimension1'  => $this->tagId(),
            'dimension2'  => $this->categoryId(),
            'dimension3'  => $this->entityContentType(),
            'dimension4'  => $this->subType(),
            'dimension6'  => $this->id(),
            'dimension7'  => $this->tagTitle(),
            'dimension8'  => $this->categoryTitle(),
            'dimension9'  => $this->authorId(),
            'dimension10' => $this->authorTitle(),
            'dimension11' => $this->userId(),
            'dimension12' => $this->userAcronym(),
            'dimension13' => $this->subSiteId,
        ];
    }

    /**
     * @param $data
     *
     * @return string
     */
    private function normalizeData($data)
    {
        return (string) $data;
    }

    /**
     * @return string
     */
    private function tagId()
    {
        if (empty($this->entity['tags'])) {
            return '';
        }

        $tag = $this->entity['tags'][0];

        return $tag['id'];

    }

    /**
     * @return string
     */
    private function categoryId()
    {
        if (empty($this->entity['categories'])) {
            return '';
        }

        $category = $this->entity['categories'][0];

        return $category['id'];
    }

    /**
     * @return mixed
     */
    private function entityContentType()
    {
        return $this->entity['selfContentType'];
    }

    /**
     * @return mixed
     */
    private function subType()
    {
        if ($this->apiVersion === 1) {
            return $this->entity['type'];
        }

        return $this->entity['type']['id'];
    }

    /**
     * @return mixed
     */
    private function id()
    {
        return $this->entity['id'];
    }

    /**
     * @return mixed|string
     * @throws \Exception
     */
    private function tagTitle()
    {
        if (empty($this->entity['tags'])) {
            return '';
        }

        $tag = $this->entity['tags'][0];

        if ($this->apiVersion === 1) {
            return $tag['lang']['title'];
        }

        $apiTag = $this->apiCall(ResourcesTypes::TAG, $tag['id']);

        return empty($apiTag['title']) ? '' : $apiTag['title'];
    }

    /**
     * @param $type
     * @param $id
     *
     * @return array
     * @throws \Exception
     */
    private function apiCall($type, $id)
    {
        $response = $this
            ->apiV2Client
            ->find($type, $id);

        return $response["statusCode"] === 200 ? $response["data"] : [];
    }

    /**
     * @return mixed|string
     * @throws \Exception
     */
    private function categoryTitle()
    {
        if (empty($this->entity['categories'])) {
            return '';
        }

        $category = $this->entity['categories'][0];

        if ($this->apiVersion === 1) {
            return $category['lang']['title'];
        }

        $apiCategory = $this->apiCall(ResourcesTypes::CATEGORY, $category['id']);

        return empty($apiCategory['title']) ? '' : $apiCategory['title'];
    }

    /**
     * @return string
     */
    private function authorId()
    {
        return empty($this->entity['author']['id']) ?
            '' :
            $this->entity['author']['id'];
    }

    /**
     * @return mixed|string
     * @throws \Exception
     */
    private function authorTitle()
    {
        if (empty($this->entity['author'])) {
            return '';
        }

        $author = $this->entity['author'];

        if ($this->apiVersion === 1) {
            return trim(sprintf('%s %s', $author['name'], $author['surname']));
        }

        $apiAuthor = $this->apiCall(ResourcesTypes::AUTHOR, $author['id']);

        return empty($apiAuthor['fullName']) ? '' : $apiAuthor['fullName'];
    }

    /**
     * @return string
     */
    private function userId()
    {
        if ($this->apiVersion === 1) {
            return empty($this->entity['userData']['id']) ?
                '':
                $this->entity['userData']['id'];
        }

        return empty($this->entity['user']['id']) ?
            '':
            $this->entity['user']['id'];
    }

    /**
     * @return string
     */
    private function userAcronym()
    {
        if ($this->apiVersion === 1) {
            return empty($this->entity['userData']['acronym']) ?
                '':
                $this->entity['userData']['acronym'];
        }

        return empty($this->entity['user']['acronym']) ?
            '':
            $this->entity['user']['acronym'];
    }
}
