<?php

namespace Comitium5\CommonWidgetsBundle\Services;

use Symfony\Component\DomCrawler\Crawler;
use Symfony\Component\Templating\EngineInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class WidgetController
 *
 * @author Daniel J. Pérez <danieljordi@bab-soft.com>
 * @package ComitiumSuite\Bundle\CSBundle\Widgets\AmpAbstract\Controller
 */
class Amp
{
    /**
     * AMP Components
     */
    const COMPONENTS_TEMPLATE_PATH = '@Comitium5CommonWidgetsBundle/Resources/views/Amp/';
    const TWEET_COMPONENT = 'twitter_amp.html.twig';
    const IMAGE_COMPONENT = 'image_amp.html.twig';
    const IFRAME_COMPONENT = 'iframe_amp.html.twig';
    const VIDEO_COMPONENT = 'video_amp.html.twig';
    const AUDIO_COMPONENT = 'audio_amp.html.twig';

    /**
     * Css Selectors
     */
    const TWITTER_SELECTOR = 'blockquote.twitter-tweet a, blockquote.twitter-video a';

    /**
     * Regex
     */
    const TWITTER_BLOCK_REGEX = '/<blockquote class="(twitter-tweet|twitter-video)"([\w\W]+?)>([\w\W]+?)<\/blockquote>/';
    const REGEX_REMOVE_SCRIPTS = '/<script([\w\W]+?)>([\w\W]+?)\/script>/';
    const IMAGE_REGEX = '/<img([\w\W]+?)\/?>/';
    const IFRAME_REGEX = '/<iframe([\w\W]+?)>([\w\W]+?)\/iframe>/';
    const VIDEO_REGEX = '/<video([\w\W]+?)>([\w\W]+?)\/video>/';
    const AUDIO_REGEX = '/<audio([\w\W]+?)>([\w\W]+?)\/audio>/';
    const HTTPS_REGEX = '/^(https:)?\/\//';
    const STYLE_REGEX = '/style="([\w\W]+?)\/?"/';

    const SCRIPT_TAG = 'script';
    const IFRAME_TAG = 'iframe';

    /**
     * @var EngineInterface
     */
    private $templateEngine;

    /**
     * Amp constructor.
     */
    public function __construct(EngineInterface $templateEngine)
    {
        $this->templateEngine = $templateEngine;
    }

    /**
     * @param $html
     * @param $request
     * @return mixed
     */
    public function htmlToAmp($html, Request $request)
    {
        $this->cleanCode($html);
        $this->twitterToAmp($html);
        $this->iframeToAmp($html);
        $this->imagesToAmp($html, $request->getSchemeAndHttpHost());
        $this->videosToAmp($html);
        $this->audioToAmp($html);

        return $html;
    }

    /**
     * @param $html
     */
    private function cleanCode(&$html)
    {
        do {
            $code = $this->getCode(
                $html,
                '<'.self::SCRIPT_TAG,
                '</'.self::SCRIPT_TAG.'>'
            );
            if (!empty($code)) {
                $html = str_replace($code, '', $html);
            }
        } while (!empty($code));

        $html = str_replace(['width="100%"', 'width=\'100%\'' ], 'width="400"', $html);
        $html = preg_replace(self::STYLE_REGEX, '', $html);
    }

    /**
     * @param $html
     */
    private function twitterToAmp(&$html)
    {
        $twitterBlocks = $this->getMatches(self::TWITTER_BLOCK_REGEX, $html);

        if (!empty($twitterBlocks)) {
            foreach ($twitterBlocks as $twitterBlock) {
                $crawler = new Crawler($twitterBlock);
                $links = $crawler->filter(self::TWITTER_SELECTOR);

                $arrayUrl = [];
                $i = 0;
                $url = $links->getNode($i);

                while (empty($arrayUrl) && !empty($url)) {
                    $url = $links->getNode($i);
                    if (empty($url)) {
                        break;
                    }
                    $url = $url->getAttribute('href');

                    if (strpos($url, 'twitter.com') !== false) {
                        $arrayUrl = explode('/', $url);
                        $twitterId = end($arrayUrl);
                        $twitterId = explode('?', $twitterId)[0];

                        if (!is_numeric($twitterId)) {
                            $arrayUrl = [];
                        }
                    }
                    $i++;
                }

                if (!empty($arrayUrl)) {
                    $twitterId = end($arrayUrl);
                    $twitterId = explode('?', $twitterId)[0];

                    $tweetComponentAmp = $this->templateEngine->render(
                        $this->componentPath(self::TWEET_COMPONENT),
                        [
                            'tweetId'    => $twitterId,
                            'blockquote' => $twitterBlock,
                            'height'     => 500,
                            'width'      => 400
                        ]
                    );

                    $html = str_replace($twitterBlock, $tweetComponentAmp, $html);
                }
            }
        }
    }

    /**
     * @param $html
     */
    private function iframeToAmp(&$html)
    {
        do {
            $code = $this->getCode(
                $html,
                '<'.self::IFRAME_TAG,
                '</'.self::IFRAME_TAG.'>'
            );
            if (!empty($code)) {
                $crawler = new Crawler($code);
                $iframeNodes = $crawler->filter('iframe');
                $iframeNode = $iframeNodes->getNode(0);

                $src = $iframeNode->getAttribute('src');
                $width = $iframeNode->getAttribute('width');
                $height = $iframeNode->getAttribute('height');

                if (preg_match(self::HTTPS_REGEX, $src)) {
                    $iframeComponent = $this->templateEngine->render(
                        $this->componentPath(self::IFRAME_COMPONENT),
                        [
                            'src'        => $src,
                            'height'     => $height,
                            'width'      => $width,
                            'attributes' => [
                                'layout' => 'responsive'
                            ]
                        ]
                    );

                    $html = str_replace($code, $iframeComponent, $html);
                } else {
                    $html = str_replace($code, '', $html);
                }
            }
        } while(!empty($code));
    }

    /**
     * @param $html
     * @param $request
     */
    private function imagesToAmp(&$html, $schema)
    {
        $images = $this->getMatches(self::IMAGE_REGEX, $html);

        if (!empty($images)) {
            foreach ($images as $image) {
                $crawler = new Crawler($image);
                $imageNodes = $crawler->filter('img');
                $nodeImage = $imageNodes->getNode(0);
                $width = $nodeImage->getAttribute('width');
                $height = $nodeImage->getAttribute('height');
                $src = $nodeImage->getAttribute('src');
                $class = $nodeImage->getAttribute('class');
                $attributes = [ 'layout' => 'responsive'];

                if ("/" == substr($src, 0, 1)) {
                    $src = $schema . $src;
                }

                if (!empty($class) && !empty(trim($class))) {
                    $attributes['class'] = $class;
                }

                $sizes = $this->calculateImageSize($src, $width, $height);

                $imageComponentAmp = $this->templateEngine->render(
                    $this->componentPath(self::IMAGE_COMPONENT),
                    [
                        'src'        => $nodeImage->getAttribute('src'),
                        'height'     => $sizes['height'],
                        'width'      => $sizes['width'],
                        'attributes' => $attributes,
                    ]
                );

                $html = str_replace($image, $imageComponentAmp, $html);
            }
        }
    }

    /**
     * @param $src
     * @param $width
     * @param $height
     * @return array
     */
    private function calculateImageSize($src, $width, $height)
    {

        if (empty($width)
            || $width == 'auto'
            || strpos($width, '%') !== false
            || empty($height) || $height == 'auto'
            || strpos($height, '%') !== false
        ) {
            list($originalWidth, $originalHeight) = getimagesize($src);
            $width = 0;
            $height = 0;

            if (!empty($originalHeight) && !empty($originalWidth)) {
                $ratio = $originalWidth / $originalHeight;

                $widthOriginal = false;
                if (empty($width) || $width == 'auto' || strpos($width, '%') !== false) {
                    $width = $originalWidth;
                    $widthOriginal = true;
                }

                $heightOriginal = false;
                if (empty($height) || $height == 'auto' || strpos($height, '%') !== false) {
                    $height = $originalHeight;
                    $heightOriginal = true;
                }

                if ($widthOriginal || $heightOriginal && !($widthOriginal && $heightOriginal)) {
                    if ($widthOriginal) {
                        $width = round($height * $ratio, 0);
                    }

                    if ($heightOriginal) {
                        $height = round($width / $ratio, 0);
                    }
                }
            }
        }

        return [
            'width'  => empty($width)  ? 400 : $width,
            'height' => empty($height) ? 400 : $height
        ];
    }

    /**
     * @param $html
     */
    private function videosToAmp(&$html)
    {
        $videos = $this->getMatches(self::VIDEO_REGEX, $html);

        if (!empty($videos)) {
            foreach ($videos as $video) {
                $crawler = new Crawler($video);
                $videoNodes = $crawler->filter('video');
                $sourceNodes = $crawler->filter('source');
                $videoNode = $videoNodes->getNode(0);
                $sourceNode = $sourceNodes->getNode(0);
                $src = $sourceNode->getAttribute('src');
                $type = $sourceNode->getAttribute('type');
                $width = $videoNode->getAttribute('width');
                $height = $videoNode->getAttribute('height');
                $poster = $videoNode->getAttribute('poster');

                if (preg_match(self::HTTPS_REGEX, $src)) {
                    $videoComponentAmp = $this->templateEngine->render(
                        $this->componentPath(self::VIDEO_COMPONENT),
                        [
                            'src'        => $src,
                            'type'       => $type,
                            'width'      => empty($width) ? 400 : $width,
                            'height'     => empty($height) ? 200 : $height,
                            'attributes' => [
                                'layout'   => 'fixed',
                                'poster'   => $poster,
                                'controls' => ''
                            ],
                        ]
                    );

                    $html = str_replace($video, $videoComponentAmp, $html);
                } else {
                    $html = str_replace($video, '', $html);
                }
            }
        }
    }

    /**
     * @param $html
     */
    private function audioToAmp(&$html)
    {
        $audios = $this->getMatches(self::AUDIO_REGEX, $html);

        if (!empty($audios)) {
            foreach ($audios as $audio) {
                $crawler = new Crawler($audio);
                $audioNodes = $crawler->filter('audio');
                $sourceNodes = $crawler->filter('source');
                $audioNode = $audioNodes->getNode(0);
                $sourceNode = $sourceNodes->getNode(0);
                $src = $sourceNode->getAttribute('src');
                $type = $sourceNode->getAttribute('type');
                $width = $audioNode->getAttribute('width');
                $height = $audioNode->getAttribute('height');
                $poster = $audioNode->getAttribute('poster');

                if (preg_match(self::HTTPS_REGEX, $src)) {
                    $audioComponentAmp = $this->templateEngine->render(
                        $this->componentPath(self::AUDIO_COMPONENT),
                        [
                            'src'        => $src,
                            'type'       => $type,
                            'width'      => empty($width) ? 400 : $width,
                            'height'     => empty($height) ? 30 : $height,
                            'attributes' => [
                                'layout'   => 'fixed',
                                'poster'   => $poster,
                                'controls' => ''
                            ],
                        ]
                    );

                    $html = str_replace($audio, $audioComponentAmp, $html);
                } else {
                    $html = str_replace($audio, '', $html);
                }
            }
        }
    }

    /**
     * @param $pattern
     * @param $html
     * @return null
     */
    private function getMatches($pattern, $html)
    {
        preg_match_all($pattern, $html, $matches);

        return !empty($matches) ? $matches[0] : null;
    }

    /**
     * @param $component
     * @return string
     */
    private function componentPath($component)
    {
        return self::COMPONENTS_TEMPLATE_PATH.$component;
    }

    /**
     * @param $text
     * @param $startTag
     * @param $endTag
     * @param int $offset
     * @return bool|string
     */
    private function getCode($text, $startTag, $endTag, &$offset = 0)
    {
        $code = '';

        if (strlen($text) > $offset) {
            $start = strpos($text, $startTag, $offset);
            if ($start !== false) {
                $end = strpos($text, $endTag, $start);
                $code = substr($text, $start, ($end-$start)+strlen($endTag));
                $offset = $end;
            }
        }

        return $code;
    }
}