<?php

namespace Comitium5\CommonWidgetsBundle\Utils\Common;

use Comitium5\ApiClientBundle\ApiClient\ResourcesTypes;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class CommonUtils
 *
 * @author Oscar Jimenez <oscarjg19.developer@gmail.com>
 * @package Comitium5\CommonWidgetsBundle\Utils\Common
 */
class CommonUtils
{
    /**
     * @var string
     */
    public static $defaultMetaDataImage = "";

    /**
     * @var string
     */
    public static $defaultMetaDataText  = "";

    /**
     * Method to get fields by typology
     *
     * @param array  $input
     * @param string $typology
     *
     * @return array
     */
    public static function getFieldsByType(array $input, $typology)
    {
        $results = [];

        foreach ($input as $value) {
            if (empty($value["field"])) {
                continue;
            }

            $field = $value["field"];
            $key   = $field["key"];

            if (preg_match("/^{$typology}_(.+)/", $key)) {
                if (!empty($value["value"])) {
                    $results[] = $value;
                }
            }
        }

        return $results;
    }

    /**
     * Method to get fields by exact key
     *
     * @param array  $input
     * @param string $keyToSearch
     *
     * @return array
     */
    public static function getFieldsByKey(array $input, $keyToSearch)
    {
        $results = [];

        foreach ($input as $value) {
            if (empty($value["field"])) {
                continue;
            }

            $field = $value["field"];
            $key   = $field["key"];

            if (preg_match("/^{$keyToSearch}$/", $key)) {
                if (!empty($value["value"])) {
                    $results[] = $value;
                }
            }
        }

        return $results;
    }

    /**
     * Method to get fields by typology
     *
     * @param array $input
     * @param $className
     *
     * @return array
     */
    public static function getRelatedContentByType(array $input, $className)
    {
        $results = [];

        foreach ($input as $value) {
            if (empty($value["relatedClass"])) {
                continue;
            }

            if (preg_match("/(.+)\\\\Entity\\\\{$className}$/", $value["relatedClass"])) {
                $results[] = $value;
            }
        }

        return $results;
    }

    /**
     * @param array $image
     * @param $sizes
     * @param bool|false $strict
     * @return array
     */
    public static function getAssetResize(array $image, $sizes, $strict = true)
    {
        if (empty($sizes)) {
            return $image;
        }

        if (empty($image['children'])) {
            return $image;
        }

        if (!is_array($sizes)) {
            $width  = $sizes;
            $height = 0;
        } else {

            if (preg_match("/^\d+\|\d+$/", $sizes[0])) {
                $sizes = explode("|",$sizes[0]);
                $width  = $sizes[0];
                $height = $sizes[1];
            } else {
                if (!isset($sizes[1])) {
                    $sizes[1] = 0;
                }
                list($width, $height) = $sizes;
            }
        }

        $prop = $width > $height ? 'width' : 'height';
        $size = $width > $height ? $width  : $height;

        foreach ($image['children'] as $children) {
            if ($strict !== true) {
                if ((int) $children["metadata"][$prop] !== (int) $size) {
                    continue;
                }
            } else {
                if ((int) $children["metadata"]["width"] !== (int) $width || (int) $children["metadata"]["height"] !== (int) $height) {
                    continue;
                }
            }

            return $children;
        }

        return $image;
    }

    /**
     * @param array $input
     * @return array
     */
    public static function getChildren(array $input)
    {
        $results = [];

        if (empty($input["children"])) {
            return $results;
        }

        foreach ($input["children"] as $child) {

            $results[] = $child["id"];

            if (!empty($child["children"])) {
                $children = self::getChildren($child);
                $results = array_merge($results, $children);
            }
        }

        return $results;
    }

    /**
     * @param array $entity
     * @return array
     */
    public static function resolveMetaData(array $entity)
    {
        $metaDataLang = [];
        $metaDataOpenGraph = [];

        // MetaData
        if (!empty($entity["lang"])) {

            $lang = $entity["lang"];

            $defaultCanonical = isset($entity["permalink"]) ? $entity["permalink"] : "";

            $metaDataLang = [
                "title" => self::resolveMetaDataSimpleValue($lang, "metaTitle"),
                "description" => self::resolveMetaDataSimpleValue($lang, "metaDescription"),
                "image" => self::resolveImageMetaData($entity),
                "canonical" => self::resolveMetaDataSimpleValue($lang, "canonical", $defaultCanonical),
                "keywords" => self::resolveMetaDataSimpleValue($lang, "metaKeywords"),
            ];

            $metaDataOpenGraph =  [
                "ogTitle" => $metaDataLang["title"],
                "ogDescription" => $metaDataLang["description"],
                "ogImage" => $metaDataLang["image"],
                "ogType" => "website",
            ];
        }

        // OpenGraph
        if (!empty($entity["openGraph"])) {

            $openGraph = $entity["openGraph"];
            $openGraphLang = $entity["openGraph"]["lang"];

            $ogImage = self::resolveMetaDataSimpleValue($openGraph, "image", isset($metaDataLang["image"]) ? $metaDataLang["image"] : "");
            $ogImageWidth = "";
            $ogImageHeight = "";

            if (is_array($ogImage)) {
                if (isset($ogImage["url"])) {
                    $ogImageWidth = !empty($ogImage["metadata"]["width"])  ? $ogImage["metadata"]["width"] : $ogImageWidth;
                    $ogImageHeight = !empty($ogImage["metadata"]["height"]) ? $ogImage["metadata"]["height"]: $ogImageHeight;
                    $ogImage = $ogImage["url"];
                }
            }

            $metaDataOpenGraph =  [
                "ogTitle" => self::resolveMetaDataSimpleValue($openGraphLang, "title", isset($metaDataLang["title"]) ? $metaDataLang["title"] : ""),
                "ogDescription" => self::resolveMetaDataSimpleValue($openGraphLang, "description", isset($metaDataLang["description"]) ? $metaDataLang["description"] : ""),
                "ogImage" => $ogImage,
                "ogImageWidth"  => $ogImageWidth,
                "ogImageHeight" => $ogImageHeight,
                "ogType" => self::resolveMetaDataSimpleValue($openGraph, "type", "website"),
            ];
        }

        return array_merge($metaDataLang,$metaDataOpenGraph);
    }

    /**
     * @param array $data
     * @param $property
     * @param string $defaultValue
     *
     * @return string
     */
    public static function resolveMetaDataSimpleValue(array $data, $property, $defaultValue = "")
    {
        $defaultValue = $defaultValue !== null ? $defaultValue : self::$defaultMetaDataText;

        return isset($data[$property]) ? $data[$property] : $defaultValue;
    }

    /**
     * @param $entity
     * @param string $defaultValue
     *
     * @return string
     */
    public static function resolveImageMetaData($entity)
    {
        $defaultValue = self::$defaultMetaDataImage;

        if (!empty($entity["selfContentType"])) {
            $contentType = $entity["selfContentType"];

            if ($contentType === ResourcesTypes::ASSETS_CODE) {
                if (!empty($entity["imageRelated"]["url"])) {
                    return $entity["imageRelated"]["url"];
                } elseif (!empty($entity["url"])) {
                    try {
                        if (getimagesize($entity["url"]) !== false) {
                            return $entity["url"];
                        }
                    }catch (\Exception $e) {
                        return $defaultValue;
                    }
                }
            }
        }


        if (!empty($entity["asset"]["url"])) {
            return $entity["asset"]["url"];
        }

        if (!empty($entity["lang"]["fieldValues"])) {
            $images = self::getFieldsByType($entity["lang"]["fieldValues"], "image");

            if (count($images)) {
                $image = $images[0];

                return empty($image["value"]["url"]) ? $defaultValue : $image["value"]["url"];
            }
        }

        return $defaultValue;
    }

    /**
     * @param array $entity
     * @return array
     */
    public static function getLoops(array $entity)
    {
        $fieldValues = empty($entity["lang"]["fieldValues"]) ? [] : $entity["lang"]["fieldValues"];

        $loops = [];

        $usedKeys = [];
        $index = 0;

        foreach ($fieldValues as $fieldValue) {

            $field = $fieldValue["field"];

            if ($field["parent"]) {

                $key = $field["key"];

                if(empty($usedKeys[$field["parent"]])) {
                    $usedKeys[$field["parent"]] = [];
                }

                if (in_array($key, $usedKeys[$field["parent"]])) {
                    $index++;
                    $usedKeys[$field["parent"]] = [$key];
                } else {
                    $usedKeys[$field["parent"]][] = $key;
                }

                $loops[$field["parent"]][$index][] = $fieldValue;
            }
        }

        return $loops;
    }

    /**
     * @param $dateTime
     * @return \DateTime
     */
    public static function instanceDateTimeFromArray($dateTime)
    {
        if (is_array($dateTime)) {
            return new \DateTime($dateTime['date'], new \DateTimeZone($dateTime['timezone']));
        }

        return $dateTime;
    }

    /**
     * @return string
     */
    public static function wordCharList()
    {
        return "ÀàÁáÈèÉéÍíÒòÓóÚúÄãËëÏïÖöÜüÇç'’-ñ";
    }

    /**
     * @param $string
     * @param $numberOfWord
     * @param string $separator
     * @param int $firstWord
     * @return mixed|string
     */
    public static function truncateWords($string, $numberOfWord, $separator = '...', $firstWord = 0)
    {
        $arrayWords = str_word_count(self::cleanText($string), 1, self::wordCharList());

        $count = count($arrayWords);

        if ($count === 0) {
            return "";
        }

        $words = implode(' ', array_splice($arrayWords, $firstWord, $numberOfWord));

        if ($count > $numberOfWord) {
            $words .= $separator;
        }

        return $words;
    }

    /**
     * @param Request $request
     * @return array|string
     */
    public static function getUserIp(Request $request)
    {
        $ip = $request->headers->get('X-Forwarded-For');

        if($ip === null) {
            $ip = $request->getClientIp();
        }

        return $ip;
    }

    /**
     * @static
     * @param  string $text
     * @return string
     */
    public static function cleanText($text)
    {
        $text = self::replaceSpecialChars($text);

        return preg_replace('/[\n\r\t\s]+/', ' ', strip_tags(
            html_entity_decode(trim($text))
        ));
    }

    /**
     * @param $input
     *
     * @return mixed
     */
    public static function replaceSpecialChars($input)
    {
        $htmlSpecialEntities = ['&nbsp;','&ldquo;','&rdquo;','&lsquo;','&rsquo;','&#63;','&#664;','&#8200;','“','”','‘','’','&#8217;','&#8220;','&#8221;','&#149;','&#183;','&#191;','&#8216;','&#8208;','&#147;','&#148;','&#145;','&#146;','&#39;'];
        $replaceHtmlSpecialEntities = [' ','"','"','\'','\'','?','·',' ','"','"','\'','\'','\'','','"','"','·','·','¿','\'','-','"','"','\'','\''];

        return str_replace($htmlSpecialEntities, $replaceHtmlSpecialEntities, $input);
    }

    /**
     * @param array $fieldValue
     * @return null|string
     */
    public static function buildFieldKeyToElastic(array $fieldValue)
    {
        $key = null;

        if (!empty($fieldValue["field"]["key"]) && !empty($fieldValue["id"])) {
            $key = sprintf("%s_%s",
                $fieldValue["id"],
                $fieldValue["field"]["key"]
            );
        }

        return $key;
    }
}
