<?php

namespace Comitium5\CommonWidgetsBundle\Search;

use Elastica\Filter\Range;
use Elastica\Query;
use Elastica\Query\Bool;
use Elastica\Query\Match;
use Elastica\Query\Nested;

/**
 * Class ElasticSearchProvider
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\CommonWidgetsBundle\Search
 */
class ElasticSearchProvider
{
    /**
     *
     */
    const QUERY_LIMIT = 10;

    /**
     *
     */
    const DEFAULT_OPERATOR = "and";

    /**
     *
     */
    const DEFAULT_TITLE_BOOST = 10;

    /**
     *
     */
    const DEFAULT_BODY_BOOST = 4;

    /**
     *
     */
    const DEFAULT_SUB_TITLE_BOOST = 5;

    /**
     *
     */
    const DEFAULT_INTRO_TITLE_BOOST = 5;

    /**
     *
     */
    const DEFAULT_NAME_BOOST = 5;

    const DEFAULT_AUTHOR_BOOST = 5;

    /**
     * @TODO remove after CS config for front project will be solved
     */
    public static $locales = [
        "es" => 1,
        "ca" => 2,
        "en" => 3,
    ];

    /**
     * @var \Elastica\Client
     */
    private $client;

    /**
     * @var string
     */
    private $index;

    /**
     * @var string
     */
    private $term;

    /**
     * @var int
     */
    private $page;

    /**
     * @var integer
     */
    private $orderType;

    /**
     * @var array
     */
    private $includedContentTypes = [];

    /**
     * @var array
     */
    private $excludedContentTypes = [];

    /**
     * @var string
     */
    private $locale;

    /**
     * @var array
     */
    private $sort = [];

    /**
     * @var int
     */
    private $limit;

    /**
     * @var array
     */
    private $subTypes = [];

    /**
     * @var string
     */
    private $localeCode;
    /**
     * @var string
     */
    private $categoriesIds;
    /**
     * @var string
     */
    private $authorIds;

    /**
     * @var
     */
    private $tags;

    /**
     * @var string
     */
    private $operator;

    /**
     * @var float
     */
    private $titleBoost;

    /**
     * @var float
     */
    private $bodyBoost;

    /**
     * @var float
     */
    private $nameBoost;

    /**
     * @var float
     */
    private $subTitleBoost;

    /**
     * @var float
     */
    private $introTitleBoost;

    /**
     * @var float
     */
    private $authorBoost;

    /**
     * @var bool
     */
    private $matchExactPhrase;

    /**
     * @var string
     */
    private $analyzer;


    /**
     * ElasticSearchProvider constructor.
     * @param $host
     * @param $port
     * @param $index
     */
    public function __construct($host, $port, $index)
    {
        $this->client = new \Elastica\Client([
            "host" => $host,
            "port" => $port,
        ]);

        $this->index = $index;
        $this->limit = self::QUERY_LIMIT;
        $this->page = 1;
        $this->operator = self::DEFAULT_OPERATOR;
        $this->titleBoost = self::DEFAULT_TITLE_BOOST;
        $this->bodyBoost = self::DEFAULT_BODY_BOOST;
        $this->introTitleBoost = self::DEFAULT_INTRO_TITLE_BOOST;
        $this->subTitleBoost = self::DEFAULT_SUB_TITLE_BOOST;
        $this->nameBoost = self::DEFAULT_NAME_BOOST;
        $this->authorBoost = self::DEFAULT_AUTHOR_BOOST;
        $this->matchExactPhrase = false;
        $this->analyzer = "frontend";
    }

    /**
     * @return \Elastica\Client
     */
    public function getClient()
    {
        return $this->client;
    }

    /**
     * @param $host
     * @param $port
     * @param null $index
     * @return $this
     */
    public function setClient($host, $port, $index = null)
    {
        $this->client = new \Elastica\Client([
            "host" => $host,
            "port" => $port,
        ]);

        if ($index !== null) {
            $this->index = $index;
        }

        return $this;
    }

    /**
     * @return array
     */
    public function search()
    {
        $index = $this->client->getIndex($this->index);

        $boolQuery = new \Elastica\Query\BoolQuery();


        $matchBoolean = new Bool();

        if (empty($this->term)) {
            $match = new \Elastica\Query\MatchAll();
            $boolQuery->addMust($match);
        } else {

            // Fields
            $match = $this->createMatchInstance();
            $match->setFieldQuery("title", $this->term);
            $match->setFieldBoost("title", $this->titleBoost);
            $match->setFieldOperator("title", $this->operator);
            $match->setFieldAnalyzer("title",$this->analyzer);
            $matchBoolean->addShould($match);

            $match = $this->createMatchInstance();
            $match->setFieldQuery("body", $this->term);
            $match->setFieldBoost("body", $this->bodyBoost);
            $match->setFieldOperator("body", $this->operator);
            $match->setFieldAnalyzer("body", $this->analyzer);
            $matchBoolean->addShould($match);

            $match = $this->createMatchInstance();
            $match->setFieldQuery("introTitle", $this->term);
            $match->setFieldBoost("introTitle", $this->introTitleBoost);
            $match->setFieldOperator("introTitle", $this->operator);
            $match->setFieldAnalyzer("introTitle", $this->analyzer);
            $matchBoolean->addShould($match);

            $match = $this->createMatchInstance();
            $match->setFieldQuery("name", $this->term);
            $match->setFieldBoost("name", $this->nameBoost);
            $match->setFieldOperator("name", $this->operator);
            $match->setFieldAnalyzer("name",$this->analyzer);
            $matchBoolean->addShould($match);

            $match = $this->createMatchInstance();
            $match->setFieldQuery("subTitle", $this->term);
            $match->setFieldBoost("subTitle", $this->subTitleBoost);
            $match->setFieldOperator("subTitle", $this->operator);
            $match->setFieldAnalyzer("subTitle", $this->analyzer);
            $matchBoolean->addShould($match);

            //Author
            $boolAuthorFieldsQuery = new Bool();

            $matchAuthorQuery = new Match();
            $matchAuthorQuery->setFieldQuery("author.title", $this->term);
            $matchAuthorQuery->setFieldBoost("author.title", $this->authorBoost);
            $matchAuthorQuery->setFieldOperator("author.title", $this->operator);
            $matchAuthorQuery->setFieldFuzziness("author.title",1);

            $nestedBoolQuery = new Query\BoolQuery();
            $nestedBoolQuery->addMust($matchAuthorQuery);

            $nestedQuery = new Nested();
            $nestedQuery->setPath("author");
            $nestedQuery->setQuery($nestedBoolQuery);

            $boolAuthorFieldsQuery->addShould($nestedQuery);
            $boolAuthorFieldsQuery->addShould($matchBoolean);

            $boolQuery->addMust($boolAuthorFieldsQuery);
        }

        # FILTER BY DATE AND ENABLED
        $rangeQuery = new Range();
        $rangeQuery->addField("publishedDate", [
            "lt" => "now",
            "time_zone" => "+01:00",
        ]);

        $boolQuery->addMust($rangeQuery->toArray());

        $termsQuery = new \Elastica\Query\Match();
        $termsQuery->setField('enabled', true);
        $boolQuery->addMust($termsQuery);

        if ($this->locale) {
            $termsQuery = new \Elastica\Query\Match();
            $termsQuery->setField('locale', $this->locale);
            $boolQuery->addMust($termsQuery);
        }

        if ($this->localeCode) {
            $termsQuery = new \Elastica\Query\Match();
            $termsQuery->setField('localeCode', $this->localeCode);
            $boolQuery->addMust($termsQuery);
        }

        if (!empty($this->categoriesIds)) {
            $termsQuery = new \Elastica\Query\Match();
            $termsQuery->setFieldAnalyzer("categoriesIds", "comma");
            $termsQuery->setField("categoriesIds", $this->categoriesIds);

            $boolQuery->addMust($termsQuery);
        }

        if (!empty($this->authorIds)) {
            $termsQuery = new \Elastica\Query\Match();
            $termsQuery->setFieldAnalyzer("authorsIds", "comma");
            $termsQuery->setField('authorsIds', $this->authorIds);

            $boolQuery->addMust($termsQuery);
        }

        if (!empty($this->tags)) {
            $termsQuery = new \Elastica\Query\Match();
            $termsQuery->setFieldAnalyzer("tags", "comma");
            $termsQuery->setField('tags', $this->tags);

            $boolQuery->addMust($termsQuery);
        }

        if (!empty($this->subTypes)) {
            $termsQuery = new \Elastica\Query\Terms();
            $termsQuery->setTerms('subType', $this->subTypes);
            $boolQuery->addMust($termsQuery);
        }

        if (!empty($this->includedContentTypes)) {
            $termsQuery = new \Elastica\Query\Terms();
            $termsQuery->setTerms('entityType', $this->includedContentTypes);
            $boolQuery->addMust($termsQuery);
        }

        if (!empty($this->excludedContentTypes)) {
            $termsQuery = new \Elastica\Query\Terms();
            $termsQuery->setTerms('entityType', $this->excludedContentTypes);
            $boolQuery->addMust($termsQuery);
        }

        $query = \Elastica\Query::create($boolQuery);
        $query->setHighlight(
            ['fields' => [
                'title' => new \stdClass()
            ]
            ]);

        if ($this->sort) {
            $query->setSort($this->sort);
        } else{
            $query->setSort([
                "_score" => ["order" => "desc" ],
                "creationDate" => ['order' => 'desc'],
            ]);
        }

        if ($this->page && $this->limit) {
            $query->setFrom(((int) $this->page - 1) * (int) $this->limit);
        }

        $results = $index->search($query, [
            "limit" => $this->limit,
        ]);

        return ElasticResultParser::parser($results);
    }

    /**
     * @param \Elastica\Query\Match|null $matchQuery
     *
     * @return \Elastica\Query\Match
     */
    public function setBoostFields(\Elastica\Query\Match $matchQuery = null)
    {
        if ($matchQuery !== null) {
            return $matchQuery;
        }

        $matchQuery = new \Elastica\Query\Match();
        $matchQuery->setFieldBoost("name", 5);
        $matchQuery->setFieldBoost("title", 10);
        $matchQuery->setFieldBoost("subTitle", 5);
        $matchQuery->setFieldBoost("body", 4);
        $matchQuery->setFieldBoost("featured", 10);
        $matchQuery->setFieldBoost("reference", 10);

        return $matchQuery;
    }

    /**
     * @return mixed
     */
    public function getTerm()
    {
        return $this->term;
    }

    /**
     * @param $term
     *
     * @return $this
     */
    public function setTerm($term)
    {
        $this->term = $term;

        return $this;
    }

    /**
     * @return mixed
     */
    public function getPage()
    {
        return $this->page;
    }

    /**
     * @param $page
     *
     * @return $this
     */
    public function setPage($page)
    {
        $this->page = $page;

        return $this;
    }

    /**
     * @return mixed
     */
    public function getOrderType()
    {
        return $this->orderType;
    }

    /**
     * @param $orderType
     *
     * @return $this
     */
    public function setOrderType($orderType)
    {
        $this->orderType = $orderType;

        return $this;
    }

    /**
     * @return mixed
     */
    public function getLocale()
    {
        return $this->locale;
    }

    /**
     * @param mixed $locale
     */
    public function setLocale($locale)
    {
        $this->locale = $locale;

        return $this;
    }

    /**
     * @return array
     */
    public function getIncludedContentTypes()
    {
        return $this->includedContentTypes;
    }

    /**
     * @param array $includedContentTypes
     */
    public function setIncludedContentTypes(array $includedContentTypes)
    {
        $this->includedContentTypes = $includedContentTypes;
        return $this;
    }

    /**
     * @return array
     */
    public function getExcludedContentTypes()
    {
        return $this->excludedContentTypes;
    }

    /**
     * @param array $excludedContentTypes
     */
    public function setExcludedContentTypes(array $excludedContentTypes)
    {
        $this->excludedContentTypes = $excludedContentTypes;
        return $this;
    }

    /**
     * @param $contentType
     */
    public function addIncludeContentType($contentType)
    {
        $this->includedContentTypes[] = $contentType;
    }

    /**
     * @param $contentType
     */
    public function addExcludedContentType($contentType)
    {
        $this->excludedContentTypes[] = $contentType;
    }

    /**
     * @return mixed
     */
    public function getSort()
    {
        return $this->sort;
    }

    /*
     * Sorting: Accept arrays as
     *  [
     *      'creationDate' => ['order' => 'desc'],
     *  ]
     *
     * @param mixed $sort
     */
    /**
     * @param $sort
     * @return $this
     */
    public function setSort(array $sort)
    {
        $this->sort = $sort;

        return $this;
    }

    /**
     * @return Match|Query\MatchPhrase
     */
    private function createMatchInstance()
    {
        if ($this->matchExactPhrase === true) {
            return new Query\MatchPhrase();
        }

        return new Match();
    }

    /**
     * @return int
     */
    public function getLimit()
    {
        return $this->limit;
    }

    /**
     * @param $limit
     * @return $this
     */
    public function setLimit($limit)
    {
        $this->limit = $limit;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getIndex()
    {
        return $this->index;
    }

    /**
     * @param $index
     *
     * @return $this
     */
    public function setIndex($index)
    {
        $this->index = $index;

        return $this;
    }

    /**
     * @return array
     */
    public function getSubTypes()
    {
        return $this->subTypes;
    }

    /**
     * @param array $subTypes
     * @return $this
     */
    public function setSubTypes(array $subTypes)
    {
        $this->subTypes = $subTypes;

        return $this;
    }

    /**
     * @return string
     */
    public function getAuthorIds()
    {
        return $this->authorIds;
    }

    /**
     * @param $authorIds
     * @return $this
     */
    public function setAuthorIds($authorIds)
    {
        $this->authorIds = $authorIds;
        return $this;
    }

    /**
     * @return string
     */
    public function getCategoriesIds()
    {
        return $this->categoriesIds;
    }

    /**
     * @param $categoriesIds
     * @return $this
     */
    public function setCategoriesIds($categoriesIds)
    {
        $this->categoriesIds = $categoriesIds;
        return $this;
    }

    /**
     * @return string
     */
    public function getLocaleCode()
    {
        return $this->localeCode;
    }

    /**
     * @param $localeCode
     * @return $this
     */
    public function setLocaleCode($localeCode)
    {
        $this->localeCode = $localeCode;

        return $this;
    }

    /**
     * @return mixed
     */
    public function getTags()
    {
        return $this->tags;
    }

    /**
     * @param $tags
     *
     * @return $this
     */
    public function setTags($tags)
    {
        $this->tags = $tags;

        return $this;
    }

    /**
     * @return string
     */
    public function getOperator()
    {
        return $this->operator;
    }

    /**
     * @param $operator
     * @return $this
     */
    public function setOperator($operator)
    {
        $this->operator = $operator;

        return $this;
    }

    /**
     * @return mixed
     */
    public static function getLocales()
    {
        return self::$locales;
    }

    /**
     * @param mixed $locales
     */
    public static function setLocales($locales)
    {
        self::$locales = $locales;
    }

    /**
     * @return float
     */
    public function getTitleBoost()
    {
        return $this->titleBoost;
    }

    /**
     * @param float $titleBoost
     */
    public function setTitleBoost($titleBoost)
    {
        $this->titleBoost = $titleBoost;
        return $this;
    }

    /**
     * @return float
     */
    public function getBodyBoost()
    {
        return $this->bodyBoost;
    }

    /**
     * @param float $bodyBoost
     */
    public function setBodyBoost($bodyBoost)
    {
        $this->bodyBoost = $bodyBoost;
        return $this;
    }

    /**
     * @return float
     */
    public function getNameBoost()
    {
        return $this->nameBoost;
    }

    /**
     * @param $nameBoost
     * @return $this
     */
    public function setNameBoost($nameBoost)
    {
        $this->nameBoost = $nameBoost;
        return $this;
    }

    /**
     * @return float
     */
    public function getSubTitleBoost()
    {
        return $this->subTitleBoost;
    }

    /**
     * @param $subTitleBoost
     * @return $this
     */
    public function setSubTitleBoost($subTitleBoost)
    {
        $this->subTitleBoost = $subTitleBoost;
        return $this;
    }

    /**
     * @return float
     */
    public function getIntroTitleBoost()
    {
        return $this->introTitleBoost;
    }

    /**
     * @param $introTitleBoost
     * @return $this
     */
    public function setIntroTitleBoost($introTitleBoost)
    {
        $this->introTitleBoost = $introTitleBoost;
        return $this;
    }

    /**
     * @return float
     */
    public function getAuthorBoost()
    {
        return $this->authorBoost;
    }

    /**
     * @param $authorBoost
     * @return $this
     */
    public function setAuthorBoost($authorBoost)
    {
        $this->authorBoost = $authorBoost;
        return $this;
    }

    /**
     * @return boolean
     */
    public function isMatchExactPhrase()
    {
        return $this->matchExactPhrase;
    }

    /**
     * @param $matchExactPhrase
     * @return $this
     */
    public function setMatchExactPhrase($matchExactPhrase)
    {
        $this->matchExactPhrase = $matchExactPhrase;

        return $this;
    }

    /**
     * @return string
     */
    public function getAnalyzer()
    {
        return $this->analyzer;
    }

    /**
     * @param string $analyzer
     * @return $this
     */
    public function setAnalyzer($analyzer)
    {
        $this->analyzer = $analyzer;

        return $this;
    }
}