<?php

namespace Comitium5\CommonWidgetsBundle\Controller;

use Comitium5\CommonWidgetsBundle\Services\AjaxRequestFactory;
use Comitium5\CommonWidgetsBundle\Services\AjaxRequestResolver;
use Comitium5\CommonWidgetsBundle\Services\SubSiteConfigLoader;
use Comitium5\CommonWidgetsBundle\ValueObject\AjaxRequestValueObject;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class AbstractWidgetControllerBase
 *
 * @author Óscar Jimenez <oscar@bab-soft.com>
 * @package Controller
 */
abstract class AbstractWidgetControllerBase extends Controller
{
    const SITE_INDEX = "_site";

    const SUBSITE_INDEX = "_subsite";

    const SUBSITE_REQUEST_KEY = "_subsite_request";

    const LOCALE_INDEX = "_locale";

    const PAGE_REQUEST_KEY = "_page_request";

    const PREVIEW_INDEX = '_preview';

    /**
     * @var array
     */
    protected static $parameters = [];

    /**
     * @var AjaxRequestValueObject
     */
    public $ajaxRequestValueObject;

    /**
     * @return array
     */
    protected static function getSite()
    {
        return isset(static::$parameters[self::SITE_INDEX]) ? static::$parameters[self::SITE_INDEX] : [];
    }

    /**
     * @return array
     */
    protected static function getSubsite()
    {
        return isset(static::$parameters[self::SUBSITE_INDEX]) ? static::$parameters[self::SUBSITE_INDEX] : [];
    }

    /**
     * @return mixed
     */
    protected static function getLocale()
    {
        if (isset(static::$parameters[self::LOCALE_INDEX])) {
            return static::$parameters[self::LOCALE_INDEX];
        }

        return null;
    }

    /**
     * @return array
     */
    public static function getParameters()
    {
        return self::$parameters;
    }

    /**
     * @param array $parameters
     */
    public static function setParameters(array $parameters)
    {
        self::$parameters = $parameters;
    }

    /**
     * @return array
     */
    protected function getPageFromRequest()
    {
        $request = $this
            ->get('request_stack')
            ->getMasterRequest();

        return $request->get(self::PAGE_REQUEST_KEY, []);
    }

    /**
     * @return array
     */
    protected function getSubsiteFromRequest()
    {
        $request = $this
            ->get('request_stack')
            ->getMasterRequest();

        return $request->get(self::SUBSITE_REQUEST_KEY, []);
    }

    /**
     * @param string $widgetClass
     * @param array $parametersMapping
     *
     * @return void
     * @throws \Exception
     */
    public function createAjaxRequestValueObject(string $widgetClass, array $parametersMapping = []): void
    {
        $factory = new AjaxRequestFactory();
        $this->ajaxRequestValueObject = $factory->create(
            $widgetClass,
            self::$parameters,
            $parametersMapping
        );
    }

    /**
     * @param string $method
     * @param array $extraParameters
     *
     * @return string
     * @throws \Exception
     */
    public function getAjaxCall(string $method, array $extraParameters = []): string
    {
        $this->ajaxRequestValueObject->validate();

        $resolver = new AjaxRequestResolver();

        $encodedParameters = $resolver->resolveEncodedParameters($this->ajaxRequestValueObject, $method, $extraParameters);

        $service = $this->ajaxRequestValueObject->getService();

        $callParameters = [
            "controller" => $this->ajaxRequestValueObject->getWidgetClass(),
            "action" => $this->ajaxRequestValueObject->getAjaxEntryPoint(),
            "method" => $method,
            "_parameters" => $encodedParameters
        ];

        $url = $this->generateUrl($service, $callParameters);

        if (self::isPreview()) {
            $homeUrl = SubSiteConfigLoader::getValue('project.subsite_home_url');
            if (str_ends_with($homeUrl, '/')) {
                $homeUrl = substr($homeUrl, 0, -1);
            }
            $url = $homeUrl . $url;
        }

        return $url;
    }

    /**
     * @param array $mapping
     *
     * @return void
     * @throws \Exception
     */
    public function resolveAjaxCall(array $mapping = []): void
    {
        $request = $this->getMasterRequest();

        $resolver = new AjaxRequestResolver();

        $parameters = $resolver->resolveDecodedParameters($request, $mapping);

        self::$parameters = $parameters;

        self::$parameters = $this->addSiteValues(self::$parameters);
    }

    /**
     * @return bool
     */
    protected static function isPreview()
    {
        if (isset(static::$parameters[self::PREVIEW_INDEX])) {
            return static::$parameters[self::PREVIEW_INDEX];
        }

        return false;
    }


    /**
     * @param array $parameters
     *
     * @return array
     * @throws \Exception
     */
    private function addSiteValues(array $parameters): array
    {
        if (empty($parameters)) {
            throw new \Exception(__METHOD__ . ": parameters can't be empty");
        }

        $parameters['_site'] = [
            'id' => SubSiteConfigLoader::getValue('project.site_id'),
            'prefix' => SubSiteConfigLoader::getValue('project.site_acronym'),
        ];

        return $parameters;
    }

    /**
     * @return Request
     */
    public function getMasterRequest(): Request
    {
        return $this->get('request_stack')->getMasterRequest();
    }
}
