<?php

namespace Comitium5\CommonWidgetsBundle\Utils\RichSnippets;

use Comitium5\CommonWidgetsBundle\Utils\Common\CommonUtils;

/**
 * Class RichSnippets
 *
 * Rich snippet helper
 *
 * @package Comitium5\CommonWidgetsBundle\Utils\RichSnippets
 */
class RichSnippets implements GoogleSnippetsInterface
{
    /**
     * @var int
     */
    public static $maxTitleLength = 110;

    /**
     * @var string
     */
    public static $defaultCompleteText = "...";

    /**
     * DefaultValues[
     *  defaultLogoUrl,
     *  defaultLogoWidth,
     *  defaultLogoHeight,
     *  imageResize,
     *  authorName,
     *  publisherName,
     * ]
     *
     * @param array $entity
     * @param array $asset
     * @param array $author
     * @param string $description
     * @param array $data
     *
     * @return array
     */
    public static function articleGoogleRichSnippet(
        array $entity,
        array $asset = [],
        array $author = [],
        $description = "",
        array $data = []
    ){
        $canonical = !empty($entity["metaData"]["canonical"]) ?
            $entity["metaData"]["canonical"] :
            strval($entity["permalink"]);

        $title = rtrim($entity['title']);

        if (mb_strlen($title) > self::$maxTitleLength) {
            $title = rtrim(mb_substr($title, 0, (self::$maxTitleLength - mb_strlen(self::$defaultCompleteText) ))).self::$defaultCompleteText;
        }

        $logo =  [
            '@type'  => self::IMAGE_OBJECT_LABEL,
            'url'    => !empty($data["defaultLogoUrl"])    ? $data["defaultLogoUrl"]    : "",
            'width'  => !empty($data["defaultLogoWidth"])  ? $data["defaultLogoWidth"]  : "",
            'height' => !empty($data["defaultLogoHeight"]) ? $data["defaultLogoHeight"] : "",
        ];

        $publishedAtDate = \DateTime::createFromFormat("Y-m-d H:i:s", $entity['publishDate']);
        $updatedAtDate   = empty($entity['modificationDate']) ?
            $publishedAtDate :
            \DateTime::createFromFormat("Y-m-d H:i:s", $entity['modificationDate']);

        $image = $logo;

        if (!empty($asset)) {
            $image = [
                '@type'  => self::IMAGE_OBJECT_LABEL,
                'url'    => $asset['url'],
                'width'  => $asset['metadata']['width'],
                'height' => $asset['metadata']['height'],
            ];
        }

        $authorName = "";

        if (!empty($data["authorName"])) {
            $authorName = $data["authorName"];
        } elseif (!empty($author["fullName"])) {
            $authorName = $author["fullName"];
        }

        $json = [
            '@context' => self::SCHEMA_ORG_URL,
            '@type'    => self::GOOGlE_ARTICLE_TYPE,
            'mainEntityOfPage' => [
                '@type' => self::WEB_PAGE_LABEL,
                '@id'   => $canonical
            ],
            'headline' => $title,
            'image'    => $image,
            'datePublished' => $publishedAtDate instanceof \DateTime ? $publishedAtDate->format(\DateTime::ISO8601) : "",
            'dateModified'  => $updatedAtDate instanceof \DateTime   ? $updatedAtDate->format(\DateTime::ISO8601) : "",
            'publisher'     => [
                '@type' => self::ORGANIZATION_LABEL,
                'name'  => !empty($data["publisherName"]) ? $data["publisherName"]: "",
                'logo'  => $logo
            ]
        ];

        if (!empty($authorName)) {
            $json['author'] = [
                '@type' => self::PERSON_LABEL,
                "name"  => $authorName,
            ];
        }

        if (!empty($description)) {
            $json['description'] = trim(CommonUtils::cleanText($description));
        }

        return $json;
    }

    /**
     * @param array $entity
     * @param string $description
     * @param array $asset
     * @param array $organizer
     * @param array $mapLocation
     *
     * @return array
     * @throws \Exception
     */
    public static function activityGoogleRichSnippet(
        array $entity,
        $description = "",
        array $asset = [],
        array $organizer = [],
        array $mapLocation = []
    ) {
        $title = rtrim($entity['title']);

        if (mb_strlen($title) > self::$maxTitleLength) {
            $title = trim(mb_substr($title, 0, (self::$maxTitleLength - mb_strlen(self::$defaultCompleteText) ))).self::$defaultCompleteText;
        }

        $description = trim(CommonUtils::cleanText($description));

        if (!empty($entity['startAt'])) {
            $startDate = \DateTime::createFromFormat("Y-m-d H:i:s", $entity['startAt']);
        } elseif (!empty($entity['publishDate'])) {
            $startDate = \DateTime::createFromFormat("Y-m-d H:i:s", $entity['publishDate']);
        } else {
            $startDate = new \DateTime();
        }

        if (!empty($entity['endAt'])) {
            $endDate = \DateTime::createFromFormat("Y-m-d H:i:s", $entity['endAt']);
        } else {
            $endDate = new \DateTime("NOW + 1 YEAR");
        }

        $url = !empty($entity["permalink"]) ? $entity["permalink"] : "";

        $locationName    = "";
        $locationAddress = "";

        if (!empty($mapLocation)) {
            $locationName    = !empty($mapLocation["title"])   ? $mapLocation["title"]   : $locationName;
            $locationAddress = !empty($mapLocation["address"]) ? $mapLocation["address"] : $locationAddress;
        }

        $location = [
            "@type"   => self::LOCATION_TYPE,
            "name"    => $locationName,
            "address" => $locationAddress,
        ];

        $image = [];

        if (!empty($asset)) {
            $image =  [
                '@type'  => self::IMAGE_OBJECT_LABEL,
                'url'    => !empty($asset["url"]) ? $asset["url"] : "",
                'width'  => !empty($asset["metadata"]["width"]) ? $asset["metadata"]["width"] : "",
                'height' => !empty($asset["metadata"]["height"])? $asset["metadata"]["height"] : "",
            ];
        }

        $offers = [
            "url"   => $url,
            "name"  => $title,
            "price" => isset($entity["free"]) && $entity["free"] === true ? 0 : (!empty($entity["price"]) ? $entity["price"] : 0),
        ];

        $performer = null;

        if (!empty($organizer["name"])) {
            $performer = $organizer["name"];
        }

        $json = [
            "@context"    => self::SCHEMA_ORG_URL,
            "@type"       => self::GOOGlE_ACTIVITY_TYPE,
            "name"        => $title,
            "description" => $description,
            "startDate"   => $startDate instanceof \DateTime ? $startDate->format(\DateTime::ISO8601) : "",
            "endDate"     => $endDate instanceof \DateTime ? $endDate->format(\DateTime::ISO8601): "",
            "url"         => $url,
            "location"    => $location,
            "offers"      => $offers,
        ];


        if (!empty($image)) {
            $json["image"] = $image;
        }

        if (!empty($performer)) {
            $json["performer"] = $performer;
        }

        return $json;
    }

    /**
     * @param $items
     * @return array
     */
    public static function itemListsRichSnippet (array $items)
    {
        return [
            "@context"        => self::SCHEMA_ORG_URL,
            "@type"           => self::ITEMS_LIST_TYPE,
            "itemListElement" => self::getJsonRelatedItems($items),
        ];
    }

    /**
     * @param $title
     * @param $description
     * @param $image
     * @param \DateTime $date
     * @param $videoUrl
     * @param $embedUrl
     * @param $duration
     * @param array $publisher
     * @return array
     */
    public static function videoSnippet(
        $title,
        $description,
        $image,
        \DateTime $date,
        $videoUrl,
        $embedUrl,
        $duration,
        array $publisher
    ) {
        return [
            '@context'     => self::SCHEMA_ORG_URL,
            '@type'        => self::VIDEO_TYPE,
            'name'         => $title,
            'description'  => $description,
            'thumbnailUrl' => $image,
            'uploadDate'   => $date->format(DATE_ISO8601),
            'publisher'    => $publisher,
            'contentUrl'   => $videoUrl,
            'embedUrl'     => $embedUrl,
            'duration'     => $duration,
        ];
    }

    /**
     * @param $publisherName
     * @param $logo
     * @param $width
     * @param $height
     * @return array
     */
    public static function publisherSnippet(
        $publisherName,
        $logo,
        $width,
        $height
    ) {
        return [
            '@type' => self::ORGANIZATION_LABEL,
            'name'  => $publisherName,
            'logo'  => [
                '@type'  => self::IMAGE_OBJECT_LABEL,
                'url'    => $logo,
                'width'  => $width,
                'height' => $height,
            ],
        ];
    }

    /**
     * @param array $items
     *
     * @return array
     */
    private static function getJsonRelatedItems(array $items)
    {
        $jsonRelatedItems = [];
        $position = 1;

        foreach ($items as $item) {
            if (empty($item['permalink'])) {
                continue;
            }

            $jsonRelatedItem = [
                "@type"    => self::ITEM_LIST_TYPE,
                "position" => $position,
                "url"      => $item['permalink'],
            ];

            $jsonRelatedItems[] = $jsonRelatedItem;
            $position++;
        }

        return $jsonRelatedItems;
    }
}
