<?php

namespace Comitium5\CommonWidgetsBundle\Twig;

use Comitium5\CommonWidgetsBundle\Services\GoogleApiService;
use Comitium5\CommonWidgetsBundle\Services\SubSiteConfigLoader;
use Comitium5\CommonWidgetsBundle\Utils\Common\CommonUtils;
use Comitium5\CommonWidgetsBundle\Utils\RichSnippets\ActivityRichSnippets;
use Comitium5\CommonWidgetsBundle\Utils\RichSnippets\ArticleRichSnippets;
use Knp\Component\Pager\PaginatorInterface;
use Twig\Environment;
use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;
use Twig\Extension\AbstractExtension;
use Twig\Extension\GlobalsInterface;
use Twig\TwigFilter;
use Twig\TwigFunction;

/**
 * Class CommonExtension
 *
 * @author Oscar Jimenez <oscarjg19.developer@gmail.com>
 * @package Comitium5\CommonWidgetsBundle\Twig
 */
class CommonExtension extends AbstractExtension implements GlobalsInterface
{
    const PAGINATE_LIMIT = 10;

    private $paginator;

    /**
     * @var string
     */
    private $isProdEnvironment;

    /**
     * @var GoogleApiService
     */
    private $googleApiService;

    public function __construct(
        PaginatorInterface $paginator,
        $isProdEnvironment,
        GoogleApiService $googleApiService
    ) {
        $this->paginator = $paginator;
        $this->isProdEnvironment = $isProdEnvironment;
        $this->googleApiService = $googleApiService;
    }

    public function getGlobals(): array {
        return [
            'is_prod_environment' => $this->isProdEnvironment
        ];
    }

    public function getFilters(): array
    {
        return [
            new TwigFilter('truncate_words', [$this, 'truncateWords']),
            new TwigFilter('html_to_text', [$this, 'htmlToText']),
        ];
    }

    /**
     * @return array
     */
    public function getFunctions()
    {
        return [
            new TwigFunction('cs_pagination_ajax', [$this, 'paginationAjax'], ['is_safe' => ['html'], 'needs_environment' => true]),
            new TwigFunction('cs_field', [$this, 'fields'], ['is_safe' => ['html'], 'needs_environment' => true]),
            new TwigFunction('cs_get_fields_by_typology', [$this, 'getFieldType']),
            new TwigFunction('cs_get_related_content_by_type', [$this, 'getRelatedContentByType']),
            new TwigFunction('cs_get_fields_by_key', [$this, 'getFieldByKey']),
            new TwigFunction('cs_asset_resize', [$this, 'getAssetResize']),
            new TwigFunction('cs_get_children', [$this, 'getChildren']),
            new TwigFunction('cs_get_loops', [$this, 'getLoops']),
            new TwigFunction('cs_article_google_rich_snippet', array($this, 'articleGoogleRichSnippet')),
            new TwigFunction('cs_activity_google_rich_snippet', array($this, 'activityGoogleRichSnippet')),
            new TwigFunction('json_decode', array($this, 'jsonDecode')),
            new TwigFunction('datetime_from_array', array($this, 'instanceDateTimeFromArray')),
            new TwigFunction('google_shorter_url', array($this, 'googleShorterUrl')),
            new TwigFunction('truncate_words', [$this, 'truncateWords']),
            new TwigFunction('html_to_text', [$this, 'htmlToText']),
            new TwigFunction('cs_config_loader_value', [$this, 'getConfigLoaderValue']),
        ];
    }

    /**
     * @return string
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError
     */
    public function paginationAjax(
        Environment $environment,
        array $apiResponse,
        $domSelectorElement,
        $controller,
        $action,
        array $parameters,
        $renderTemplate = null,
        $paginationTemplate = null
    ) {
        $pagination = $this->paginator->paginate(
            $apiResponse['results'],
            $apiResponse['page'],
            empty($apiResponse['limit']) ? self::PAGINATE_LIMIT : $apiResponse['limit']
        );

        $pagination->setTotalItemCount($apiResponse['total']);

        return $environment->render(
            $renderTemplate ?: 'Comitium5CommonWidgetsBundle:Paginate:pagination_ajax.html.twig', [
                'pagination' => $pagination,
                'domSelector' => $domSelectorElement,
                'controller' => $controller,
                'action' => $action,
                'parameters' => \json_encode($parameters),
                'page' => $apiResponse['page'],
                'pages' => $apiResponse['pages'],
                'template' => $paginationTemplate ?: 'Comitium5CommonWidgetsBundle:Paginate:theme/bootstrap_v3_pagination.html.twig',
            ]
        );
    }

    /**
     * @return string
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError
     */
    public function fields(Environment $environment, $type, $name, $content, $id = '')
    {
        return $environment->render(
            'Comitium5CommonWidgetsBundle:Fields:visual_editor_field.html.twig', [
                'type' => $type,
                'name' => $name,
                'content' => $content,
                'id' => $id
            ]
        );
    }

    /**
     * @param array $input
     * @param $typology
     *
     * @return array
     */
    public function getFieldType(array $input, $typology)
    {
        return CommonUtils::getFieldsByType($input, $typology);
    }

    /**
     * @param array $input
     * @param $key
     *
     * @return array
     */
    public function getFieldByKey(array $input, $key)
    {
        return CommonUtils::getFieldsByKey($input, $key);
    }

    /**
     * @param array $input
     * @param $className
     *
     * @return array
     */
    public function getRelatedContentByType(array $input, $className)
    {
        return CommonUtils::getRelatedContentByType($input, $className);
    }

    /**
     * @param array $asset
     * @param $sizes
     * @param bool|true $strict
     * @return array
     */
    public function getAssetResize(array $asset, $sizes, $strict = true)
    {
        return CommonUtils::getAssetResize($asset, $sizes, $strict);
    }

    /**
     * @param array         $asset
     * @param integer|array $sizes
     *
     * @return array
     */
    public function getChildren(array $input)
    {
        return CommonUtils::getChildren($input);
    }

    /**
     * @param array $entity
     * @return array
     */
    public function getLoops(array $entity)
    {
        return CommonUtils::getLoops($entity);
    }

    /**
     * @param array $dateTime
     * @return \DateTime
     */
    public function instanceDateTimeFromArray(array $dateTime)
    {
        return CommonUtils::instanceDateTimeFromArray($dateTime);
    }

    /**
     * @param array $article
     * @param array $defaultValues
     *
     * @return array
     */
    public function articleGoogleRichSnippet(array $article, array $defaultValues = [])
    {
        return ArticleRichSnippets::googleRichSnippet($article, $defaultValues);
    }

    /**
     * @param array $activity
     * @return array
     */
    public function activityGoogleRichSnippet(array $activity)
    {
        return ActivityRichSnippets::googleRichSnippet($activity);
    }

    /**
     * @param $str
     * @param bool $array
     * @return mixed
     */
    public function jsonDecode($str, $array = true)
    {
        return json_decode($str, $array);
    }

    /**
     * @param $string
     * @param $numberOfWord
     * @param string $separator
     * @param int $firstWord
     * @return mixed|string
     */
    public function truncateWords($string, $numberOfWord, $separator = '...', $firstWord = 0)
    {
        return CommonUtils::truncateWords($string, $numberOfWord, $separator, $firstWord);
    }

    /**
     * @param $string
     * @return string
     */
    public function htmlToText($string)
    {
        return CommonUtils::cleanText($string);
    }

    /**
     * @param $key
     * @param array $subSite
     *
     * @return mixed|null
     * @throws \Exception
     */
    public function getConfigLoaderValue($key, array $subSite = [])
    {
        return SubSiteConfigLoader::getValue($key, $subSite);
    }

    /**
     * @param $url
     * @param bool $useCache
     * @param int $ttl
     * @return mixed|null|string
     */
    public function googleShorterUrl($url, $useCache = true, $ttl = GoogleApiService::DEFAULT_TTL)
    {
        return $this
            ->googleApiService
            ->generateShortedUrl($url, $useCache, $ttl);
    }

    /**
     * Returns the name of the extension.
     *
     * @return string The extension name
     */
    public function getName(): string
    {
        return "comitium5_common_widgets_bundle_pagination_twig_extension";
    }
}
