<?php

namespace Comitium5\CommonWidgetsBundle\Twig;

use Comitium5\CommonWidgetsBundle\Utils\Assets\AssetsUtils;
use Comitium5\CommonWidgetsBundle\Utils\Assets\VideoUtils;
use Twig\Environment;
use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Class AssetsExtension
 *
 * @author Oscar Jimenez <oscarjg19.developer@gmail.com>
 * @package Comitium5\CommonWidgetsBundle\Twig
 */
class AssetsExtension extends AbstractExtension
{
    /**
     * @var string
     */
    private $assetsUtils;

    /**
     * @var VideoUtils
     */
    private $videoUtils;

    /**
     * @var int
     */
    private $mimeTypeTimeout;

    /**
     * AssetsExtension constructor.
     * @param VideoUtils $videoUtils
     * @param AssetsUtils $assetsUtils
     * @param int $mimeTypeTimeout
     */
    public function __construct(
        VideoUtils $videoUtils,
        AssetsUtils $assetsUtils,
        $mimeTypeTimeout = 2
    ) {
        $this->videoUtils       = $videoUtils;
        $this->assetsUtils      = $assetsUtils;
        $this->mimeTypeTimeout  = $mimeTypeTimeout;
    }

    public function getFunctions(): array
    {
        return [
            new TwigFunction(
                'cs_asset_url',
                [$this, 'buildAssetUrl']
            ),
            new TwigFunction(
                'cs_asset_image_url',
                [$this, 'buildAssetUrl']
            ),
            new TwigFunction(
                'cs_asset_video_url',
                [$this, 'buildAssetUrl']
            ),
            new TwigFunction(
                'cs_asset_video_is_iframe',
                [$this, 'isVideoIframe']
            ),
            new TwigFunction(
                'cs_asset_audio_render',
                [$this, 'audioRender'],
                ['is_safe' => ['html'],'needs_environment' => true]
            ),
            new TwigFunction(
                'cs_asset_video_render',
                [$this, 'videoRender'],
                ['is_safe' => ['html'],'needs_environment' => true]
            ),
            new TwigFunction(
                'cs_pro_src_set',
                [$this, 'generateProSrcSet']
            ),
            new TwigFunction(
                'cs_get_mime_type',
                [$this, 'getMimeType']
            ),
            new TwigFunction(
                'children_match_crops',
                [$this, 'getChildrenMatchCrops']
            ),
            new TwigFunction(
                'children_match_crops_and_sizes',
                [$this, 'getChildrenMatchCropsAndSizes']
            ),
        ];
    }

    public function getFilters(): array
    {
        return [];
    }

    /**
     * @param $url
     *
     * @return string
     */
    public function buildAssetUrl($url)
    {
        return $this->assetsUtils->buildAssetUrl($url);
    }

    /**
     * @param $embedCode
     *
     * @return bool
     */
    public function isVideoIframe($embedCode)
    {
        return $this->videoUtils->isIframe($embedCode);
    }

    /**
     * @return string
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError
     */
    public function audioRender(Environment $environment, $url, $class = "")
    {
        return $environment->render(
            'Comitium5CommonWidgetsBundle:Assets:Audio/render_standard.html.twig', [
                'class' => $class,
                'url' => $url,
            ]
        );
    }

    /**
     * @return string
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError
     */
    public function videoRender(Environment $environment, $url, $class = "")
    {
        return $environment->render(
            'Comitium5CommonWidgetsBundle:Assets:Video/render_standard.html.twig', [
                'class' => $class,
                'url' => $url,
            ]
        );
    }

    /**
     * @param array $asset
     * @param array $matchCrops
     * @return string
     */
    public function generateProSrcSet(array $asset, array $matchCrops = [])
    {
        return AssetsUtils::generateProSrcSet($asset,$matchCrops);
    }

    /**
     * @param $url
     * @return string
     */
    public function getMimeType($url)
    {
        return AssetsUtils::getMimeType(
            $url,
            $this->mimeTypeTimeout
        );
    }

    /**
     * @param array $asset
     * @param array $matchCrops
     * @return array
     */
    public function getChildrenMatchCrops(array $asset, array $matchCrops)
    {
        return AssetsUtils::getChildrenMatchCrops($asset,$matchCrops);
    }

    /**
     * @param array $asset
     * @param array $matchCrops
     * @return array
     */
    public function getChildrenMatchCropsAndSizes(array $asset, array $matchCrops): array
    {
        return AssetsUtils::getChildrenMatchCropsAndSizes($asset,$matchCrops);
    }

    /**
     * Returns the name of the extension.
     *
     * @return string The extension name
     */
    public function getName(): string
    {
        return "comitium5_common_widgets_bundle_asset_twig_extension";
    }
}
