<?php

namespace Comitium5\CommonWidgetsBundle\PayWall\Security;

use Comitium5\CommonWidgetsBundle\PayWall\ValueObject\PayWallObject;
use Comitium5\CommonWidgetsBundle\Services\Security\ComitiumUser;

/**
 * Class PayWallVoter
 *
 * @author Óscar Jimenez <oscar@bab-soft.com>
 * @package Comitium5\CommonWidgetsBundle\PayWall\Security
 */
class PayWallVoter
{
    /**
     * @throws \Exception
     */
    public function hasAccess(PayWallObject $payWallObject, ComitiumUser $user = null, string $source = ""): bool
    {
        if ($source === PayWallObject::GOOGLE_SOURCE) {
            return true;
        }

        $payWallOptions = $payWallObject->getPayWallOptions();

        if (empty($payWallOptions['type'])) {
            return false;
        }

        if ($payWallOptions['type'] === PayWallObject::PAYWALL_HARD_TYPE) {
            return $this->handleHard($payWallObject, $user);
        }

        if ($payWallOptions['type'] === PayWallObject::PAYWALL_COUNTER_TYPE) {
            return $this->handleCounter($payWallObject, $user);
        }

        if ($payWallOptions['type'] === PayWallObject::PAYWALL_DATE_TYPE) {
            return $this->handleDate($payWallObject, $user);
        }

        //@TODO Add more paywall types here if is needed

        return false;
    }

    /**
     * @throws \Exception
     */
    private function handleHard(PayWallObject $payWallObject, ComitiumUser $user = null): bool
    {
        $objectSubscriptions = $payWallObject->getObjectSubscriptions();
        $userSubscriptions   = empty($user) ? [] : $user->getSubscriptions();

        if (count($objectSubscriptions) === 0) {
            return true;
        }

        return $this->hasUserSubscriptionAccess($userSubscriptions, $objectSubscriptions);
    }

    /**
     * @throws \Exception
     */
    private function handleCounter(PayWallObject $payWallObject, ComitiumUser $user = null): bool
    {
        $payWallOptions = $payWallObject->getPayWallOptions();

        /**
         * If user is has been subscribed return true (allowed)
         */
        if ($this->handleHard($payWallObject, $user) === true) {
            return true;
        }

        if (empty($payWallOptions['data']['counterLimit'])) {
            throw new \Exception('counterLimit option must be provided under data for counter paywall type');
        }

        $counterLimit = $payWallOptions['data']['counterLimit'];
        $userCookies  = $user->getCookies();

        if (empty($userCookies['comitium-paywall-counter'])) {
            return false;
        }

        $visited = explode(',', $userCookies['comitium-paywall-counter']);

        return count($visited) <= (int) $counterLimit;
    }

    /**
     * @throws \Exception
     */
    private function handleDate(PayWallObject $payWallObject, ComitiumUser $user = null): bool
    {
        $payWallOptions = $payWallObject->getPayWallOptions();

        /**
         * If user is has been subscribed return true (allowed)
         */
        if ($this->handleHard($payWallObject, $user) === true) {
            return true;
        }

        if (empty($payWallOptions['data']['dateLimit'])) {
            throw new \Exception('dateLimit option must be provided under data for date paywall type');
        }

        $dateLimit = $payWallOptions['data']['dateLimit'];
        $dateFormat = $payWallOptions['data']['dateFormat'] ?? "d/m/Y H:i";

        $currentDate = new \DateTime();
        $limitDate = (\DateTime::createFromFormat($dateFormat, $dateLimit));

        if ($limitDate === false) {
            return false;
        }

        return $limitDate < $currentDate;
    }

    /**
     * @param array $userSubscriptions
     * @param array $objectSubscriptions
     *
     * @return bool
     * @throws \Exception
     */
    private function hasUserSubscriptionAccess(array $userSubscriptions, array $objectSubscriptions)
    {
        $now = (new \DateTime('now'))->format("Y-m-d H:i:s");

        $intersection = array_filter($userSubscriptions, function ($userSubscription) use ($objectSubscriptions, $now) {
            foreach ($objectSubscriptions as $objectSubscription) {
                $isValidSubscription = is_null($userSubscription['expirationDate']) || $userSubscription['expirationDate'] > $now;

                if ($objectSubscription['id'] === $userSubscription['id'] && $isValidSubscription) {
                    return true;
                }
            }

            return false;
        });

        return count($intersection) > 0;
    }
}
