<?php

namespace Comitium5\ApiClientBundle\Client;

use Comitium5\ApiClientBundle\Parser\ResponseParser;
use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\Exception\RequestException;
use Comitium5\ApiClientBundle\Normalizer\RequestParametersNormalizer\GuzzleJsonParametersNormalizer;
use Comitium5\ApiClientBundle\Normalizer\RequestParametersNormalizer\GuzzleQueryParametersNormalizer;
use Comitium5\ApiClientBundle\Normalizer\RequestParametersNormalizer\RequestParametersNormalizerInterface;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Class RestClient
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\ApiClientBundle\Client
 */
abstract class RestClient implements RestInterface
{
    /**
     * @var string
     */
    protected $endPoint;

    /**
     * @var string
     */
    protected $token;

    /**
     * @var GuzzleClient
     */
    protected $client;

    /**
     * @var bool
     */
    protected $notFromCache;

    /**
     * RestClient constructor.
     *
     * @param $endPoint
     * @param $token
     * @param bool $verify
     * @param bool $httpErrors
     */
    public function __construct($endPoint, $token, $verify = false, $httpErrors = true)
    {
        $this->endPoint     = $endPoint;
        $this->token        = $token;
        $this->client       = new GuzzleClient([
            'verify'      => $verify,
            'http_errors' => $httpErrors,
            'base_uri'    => $this->endPoint,
        ]);
        $this->notFromCache = false;
    }

    /**
     * @param $url
     * @param array $parameters
     * @return array
     * @throws \GuzzleHttp\Exception\GuzzleException|\TypeError
     */
    public function get($url, $parameters = [])
    {
        $parameters = $this->isValidParameters($parameters,GuzzleQueryParametersNormalizer::class);

        return $this
            ->sendRequest(
                Request::METHOD_GET,
                $url,
                $parameters->normalize()
            );
    }

    /**
     * @param $url
     * @param array|RequestParametersNormalizerInterface $parameters
     * @return array
     * @throws \GuzzleHttp\Exception\GuzzleException|\TypeError
     */
    public function post($url, $parameters = [])
    {
        $parameters = $this->isValidParameters($parameters,GuzzleJsonParametersNormalizer::class);

        return $this
            ->sendRequest(
                Request::METHOD_POST,
                $url,
                $parameters->normalize()
            );
    }

    /**
     * @param $url
     * @param array|RequestParametersNormalizerInterface $parameters
     * @return array
     * @throws \GuzzleHttp\Exception\GuzzleException|\TypeError
     */
    public function put($url, $parameters = [])
    {
        $parameters = $this->isValidParameters($parameters,GuzzleJsonParametersNormalizer::class);

        return $this
            ->sendRequest(
                Request::METHOD_PUT,
                $url,
                $parameters->normalize()
            );
    }

    /**
     * @param $url
     * @return array|mixed
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function delete($url)
    {
        return $this
            ->sendRequest(
                Request::METHOD_DELETE,
                $url,
                []
            );
    }

    /**
     * @return $this
     */
    public function notFromCache()
    {
        $this->notFromCache = true;

        return $this;
    }

    /**
     * @return $this
     */
    public function fromCache()
    {
        $this->notFromCache = false;

        return $this;
    }

    /**
     * @param $method
     * @param $url
     * @param $params
     * @return array
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    protected function sendRequest($method, $url, $params)
    {
        if (!$this->client instanceof GuzzleClient) {
            throw new \Exception("Api client must be set before create request");
        }

        try {
            $params["headers"] = $this->buildRequestHeaders($params);

            $response = $this
                ->client
                ->request(
                    $method,
                    $url,
                    $params
                );

            return $this->handleResponse($response);
        } catch (RequestException $e) {
            $response = $e->getResponse();

            if ($response === null) {
                $response = new \GuzzleHttp\Psr7\Response(Response::HTTP_BAD_REQUEST, [], $e->getMessage());
            }

            return $this->buildErrorResponse($response);
        }
    }

    /**
     * @param ResponseInterface $response
     * @return array
     */
    protected function buildErrorResponse(ResponseInterface $response)
    {
        try {
            $responseData = ResponseParser::decodeResponse($response);
        } catch (\RuntimeException $e) {
            $responseData = null;
        }

        return [
            "statusCode" => $response->getStatusCode(),
            "message"    => $response->getReasonPhrase(),
            "response"   => $responseData,
            "url"        => ""
        ];
    }

    /**
     * @param ResponseInterface $response
     * @return array
     */
    protected function handleResponse(ResponseInterface $response)
    {
        if ($response->getStatusCode() !== Response::HTTP_OK) {
            return $this
                ->buildErrorResponse($response);
        }

        $statusCode = $response->getStatusCode();
        $data = ResponseParser::decodeResponse($response);

        if (isset($data["results"])) {
            if (empty($data["results"])) {
                $statusCode = Response::HTTP_NO_CONTENT;
            }
        }

        return [
            "statusCode" => $statusCode,
            "message"    => $response->getReasonPhrase(),
            "data"       => $data,
        ];
    }

    /**
     * @param $params
     * @return mixed
     */
    protected function buildRequestHeaders($params)
    {
        $defaultHeaders = $this
            ->client
            ->getConfig("headers");

        $paramHeaders = isset($params["headers"]) ?
            $params["headers"]:
            [];

        if ($this->notFromCache === true) {
            $defaultHeaders["Cache-Control"] = "no-cache";
        }

        return array_merge($defaultHeaders, $paramHeaders);
    }

    /**
     * @param $parametersArrayOrClass
     * @param $defaultClass
     *
     * @return mixed
     */
    protected function isValidParameters($parametersArrayOrClass, $defaultClass)
    {
        if ($parametersArrayOrClass instanceof RequestParametersNormalizerInterface) {
            return $parametersArrayOrClass;
        }

        if (is_array($parametersArrayOrClass)) {
            /** @var $defaultClass RequestParametersNormalizerInterface */
            return new $defaultClass($parametersArrayOrClass);
        }

        throw new \TypeError("Parameters are not valid");
    }
}
