<?php

namespace Comitium5\ApiClientBundle\Client;

use Comitium5\ApiClientBundle\Parser\ResponseParser;
use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\Exception\RequestException;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Class RestClient
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\ApiClientBundle\Client
 */
abstract class RestClient implements RestInterface
{
    /**
     * @var string
     */
    protected $endPoint;

    /**
     * @var string
     */
    protected $token;

    /**
     * @var GuzzleClient
     */
    protected $client;

    /**
     * @var bool
     */
    protected $notFromCache;

    /**
     * Client constructor.
     * @param string $endPoint
     * @param string $token
     */
    public function __construct($endPoint, $token)
    {
        $this->endPoint = $endPoint;
        $this->token    = $token;
        $this->client   =  new GuzzleClient([
            'base_uri' => $this->endPoint
        ]);
        $this->notFromCache = false;
    }

    /**
     * @param $url
     * @param array $parameters
     * @return array
     * @throws \Exception
     */
    public function get($url, array $parameters = [])
    {
        return $this
            ->sendRequest(
                Request::METHOD_GET,
                UrlResolver::fixUrl($url),
                [
                    "query" => $parameters
                ]
            );
    }

    /**
     * @param $url
     * @param array $parameters
     * @return array
     * @throws \Exception
     */
    public function post($url, array $parameters = [])
    {
        return $this
            ->sendRequest(
                Request::METHOD_POST,
                UrlResolver::fixUrl($url),
                [
                    "body" => \GuzzleHttp\json_encode($parameters)
                ]
            );
    }

    /**
     * @param $url
     * @param array $parameters
     * @return array
     * @throws \Exception
     */
    public function put($url, array $parameters = [])
    {
        return $this
            ->sendRequest(
                Request::METHOD_PUT,
                UrlResolver::fixUrl($url),
                [
                    "body" => \GuzzleHttp\json_encode($parameters)
                ]
            );
    }

    /**
     * @param $url
     * @return array|mixed
     * @throws \Exception
     */
    public function delete($url)
    {
        return $this
            ->sendRequest(
                Request::METHOD_DELETE,
                UrlResolver::fixUrl($url),
                []
            );
    }

    /**
     * @return $this
     */
    public function notFromCache()
    {
        $this->notFromCache = true;

        return $this;
    }

    /**
     * @return $this
     */
    public function fromCache()
    {
        $this->notFromCache = false;

        return $this;
    }

    /**
     * @param $method
     * @param $url
     * @param $params
     * @return array
     * @throws \Exception
     */
    protected function sendRequest($method, $url, $params)
    {
        if (!$this->client instanceof GuzzleClient) {
            throw new \Exception("Api client must be set before create request");
        }

        try {
            $params["headers"] = $this->buildRequestHeaders($params);

            $response = $this
                ->client
                ->request(
                    $method,
                    $url,
                    $params
                );

            return $this->handleResponse($response);
        } catch (RequestException $e) {
            return $this
                ->buildErrorResponse($e->getResponse());
        }
    }

    /**
     * @param ResponseInterface $response
     * @return array
     */
    protected function buildErrorResponse(ResponseInterface $response)
    {
        try {
            $responseData = ResponseParser::decodeResponse($response);
        } catch (\RuntimeException $e) {
            $responseData = null;
        }

        return [
            "statusCode" => $response->getStatusCode(),
            "message"    => $response->getReasonPhrase(),
            "response"   => $responseData,
            "url"        => ""
        ];
    }

    /**
     * @param ResponseInterface $response
     * @return array
     */
    protected function handleResponse(ResponseInterface $response)
    {
        if ($response->getStatusCode() !== Response::HTTP_OK) {
            return $this
                ->buildErrorResponse($response);
        }

        $statusCode = $response->getStatusCode();
        $data = ResponseParser::decodeResponse($response);

        if (isset($data["results"])) {
            if (empty($data["results"])) {
                $statusCode = Response::HTTP_NO_CONTENT;
            }
        }

        return [
            "statusCode" => $statusCode,
            "message"    => $response->getReasonPhrase(),
            "data"       => $data,
        ];
    }

    /**
     * @param $params
     * @return mixed
     */
    protected function buildRequestHeaders($params)
    {
        $defaultHeaders = $this
            ->client
            ->getConfig("headers");

        $paramHeaders = isset($params["headers"]) ?
            $params["headers"]:
            [];

        if ($this->notFromCache === true) {
            $defaultHeaders["Cache-Control"] = "no-cache";
        }

        return array_merge($defaultHeaders, $paramHeaders);
    }
}