<?php

namespace Comitium5\ApiClientBundle\Client;

use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Message\ResponseInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Class RestClient
 *
 * @author Óscar Jiménez <oscarjg19.developer@gmail.com>
 * @package Comitium5\ApiClientBundle\Client
 */
abstract class RestClient implements RestInterface
{
    /**
     * @var string
     */
    protected $endPoint;

    /**
     * @var string
     */
    protected $token;

    /**
     * @var GuzzleClient
     */
    protected $client;

    /**
     * Client constructor.
     * @param string $endPoint
     * @param string $token
     */
    public function __construct($endPoint, $token)
    {
        $this->endPoint = $endPoint;
        $this->token    = $token;
        $this->client   =  new GuzzleClient([
            'base_url' => $this->endPoint
        ]);
    }

    /**
     * @param $url
     * @param array $parameters
     * @return array
     */
    public function get($url, array $parameters = [])
    {
        return $this
            ->sendRequest(
                Request::METHOD_GET,
                UrlResolver::fixUrl($url),
                [
                    "query" => $parameters
                ]
            );
    }

    /**
     * @param $url
     * @param array $parameters
     * @return array
     */
    public function post($url, array $parameters = [])
    {
        return $this
            ->sendRequest(
                Request::METHOD_POST,
                UrlResolver::fixUrl($url),
                [
                    "body" => $parameters
                ]
            );
    }

    /**
     * @param $url
     * @param array $parameters
     * @return array
     */
    public function put($url, array $parameters = [])
    {
        return $this
            ->sendRequest(
                Request::METHOD_PUT,
                UrlResolver::fixUrl($url),
                [
                    "body" => $parameters
                ]
            );
    }

    /**
     * @param $url
     * @return array
     */
    public function delete($url)
    {
        return $this
            ->sendRequest(
                Request::METHOD_DELETE,
                UrlResolver::fixUrl($url),
                []
            );
    }

    /**
     * @param $method
     * @param $url
     * @param $params
     * @return array
     * @throws \Exception
     */
    protected function sendRequest($method, $url, $params)
    {
        if (!$this->client instanceof GuzzleClient) {
            throw new \Exception("Api client must be set before create request");
        }

        $request = $this
            ->client
            ->createRequest(
                $method,
                $url,
                $params
            );

        try {
            $response = $this
                ->client
                ->send($request);

            return $this->handleResponse($response);
        } catch (RequestException $e) {
            return $this
                ->buildErrorResponse($e->getResponse());
        }
    }

    /**
     * @param ResponseInterface $response
     * @return array
     */
    protected function buildErrorResponse(ResponseInterface $response)
    {
        try {
            $responseData = $response->json();
        } catch (\RuntimeException $e) {
            $responseData = null;
        }

        return [
            "statusCode" => $response->getStatusCode(),
            "message"    => $response->getReasonPhrase(),
            "response"   => $responseData,
            "url"        => $response->getEffectiveUrl()
        ];
    }

    /**
     * @param ResponseInterface $response
     * @return array
     */
    protected function handleResponse(ResponseInterface $response)
    {
        if ($response->getStatusCode() !== Response::HTTP_OK) {
            return $this
                ->buildErrorResponse($response);
        }

        return [
            "statusCode" => $response->getStatusCode(),
            "message"    => $response->getReasonPhrase(),
            "data"       => $response->json(),
        ];
    }
}