<?php

namespace Comitium5\ApiClientBundle\Provider;

use Comitium5\ApiClientBundle\ApiClient\ClientInterface;
use GuzzleHttp\Message\Response;

class ApiProvider
{
    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @param ClientInterface $client
     */
    public function __construct(ClientInterface $client)
    {
        $this->client = $client;
    }

    /**
     * @param $type
     * @param $id
     * @return mixed|void
     * @throws \Exception
     */
    public function find($type, $id)
    {
        if (is_array($id)) {
            if (!isset($id["id"])) {
                throw new \Exception("Id should be an integer or an array with id key");
            }

            $id = $id["id"];
        }

        $response = $this->client->get("{type}/{id}", array(
            'type' => $type,
            'id'   => $id,
        ));

        if ($this->isSuccessful($response) && $this->isJSON($response)) {
            return $response->json();
        }

        return;
    }

    /**
     * @param $type
     * @param  array       $parameters
     * @return array|mixed
     */
    public function findBy($type, array $parameters = [])
    {
        $path = "{type}";
        $pathParams = array('type' => $type);

        if (array_key_exists('id', $parameters)) {
            $path .= "/{id}";
            $pathParams['id'] = $parameters['id'];
            unset($parameters['id']);
        } elseif (array_key_exists('slug', $parameters)) {
            $path .= "/{slug}";
            $pathParams['slug'] = $parameters['slug'];
            unset($parameters['slug']);
        }

        $response = $this->client->get($path, $pathParams, array(
            'query' => $parameters,
        ));

        if ($this->isSuccessful($response) && $this->isJSON($response)) {
            return $response->json();
        }

        return;
    }

    /**
     * @param $type
     * @param $id
     *
     * @return mixed|void
     */
    public function remove($type, $id)
    {
        $response = $this->client->delete("{type}/{id}", [
            'type' => $type,
            'id' => $id,
        ]);

        if ($this->isSuccessful($response) && $this->isJSON($response)) {
            return $response->json();
        }

        return;
    }

    /**
     * @param $type
     * @param $parameters
     *
     * @return mixed|void
     */
    public function removeBy($type, $parameters)
    {
        $path = "{type}";
        $pathParams = array('type' => $type);

        if (array_key_exists('id', $parameters)) {
            $path .= "/{id}";
            $pathParams['id'] = $parameters['id'];
            unset($parameters['id']);
        } elseif (array_key_exists('slug', $parameters)) {
            $path .= "/{slug}";
            $pathParams['slug'] = $parameters['slug'];
            unset($parameters['slug']);
        }

        $response = $this->client->delete($path, $pathParams, array(
            'query' => $parameters,
        ));

        if ($this->isSuccessful($response) && $this->isJSON($response)) {
            return $response->json();
        }

        return;
    }

    /**
     * @param  Response $response
     * @return bool
     */
    public function isSuccessful(Response $response)
    {
        return ($response->getStatusCode() >= 200 && $response->getStatusCode() < 300) || $response->getStatusCode() == 304;
    }

    /**
     * @param  Response $response
     * @return bool
     */
    public function isJSON(Response $response)
    {
        return 'application/json' == $response->getHeader('Content-Type');
    }

    /**
     * {@inheritdoc}
     */
    public function call($method, $url, array $parameters = array(), array $requestConfig = array())
    {
        return $this->client->call($method, $url, $parameters, $requestConfig);
    }

    /**
     * {@inheritdoc}
     */
    public function get($url, array $parameters = array(), array $requestConfig = array())
    {
        return $this->client->get($url, $parameters, $requestConfig);
    }

    /**
     * {@inheritdoc}
     */
    public function put($url, array $parameters = array(), array $requestConfig = array())
    {
        return $this->client->put($url, $parameters, $requestConfig);
    }

    /**
     * {@inheritdoc}
     */
    public function post($url, array $parameters = array(), array $requestConfig = array())
    {
        return $this->client->post($url, $parameters, $requestConfig);
    }

    /**
     * {@inheritdoc}
     */
    public function patch($url, array $parameters = array(), array $requestConfig = array())
    {
        return $this->client->patch($url, $parameters, $requestConfig);
    }

    /**
     * {@inheritdoc}
     */
    public function delete($url, array $parameters = array(), array $requestConfig = array())
    {
        return $this->client->delete($url, $parameters, $requestConfig);
    }

    /**
     * @param $code
     */
    public function setAcceptLanguage($code)
    {
        $this->client->setAcceptLanguage($code);
    }

    /**
     * @param $forceContent
     */
    public function setForceGetContent($forceContent)
    {
        $this->client->setForceGetContent($forceContent);
    }

    /**
     * @param boolean $useCache
     */
    public function useCache($useCache)
    {
        $this->client->setUseCache($useCache);
    }

    /**
     * @return ClientInterface
     */
    public function getClient()
    {
        return $this->client;
    }

    /**
     * @param ClientInterface $client
     * @return $this
     */
    public function setClient($client)
    {
        $this->client = $client;

        return $this;
    }

    /**
     * @param array $options
     * @return ClientInterface
     */
    public function buildClient(array $options = [])
    {
        $endpoint = !empty($options["endpoint"]) ? $options["endpoint"] : $this->client->getEndPoint();
        $version = !empty($options["version"]) ? $options["version"] : $this->client->getVersion();
        $site = !empty($options["site"]) ? $options["site"] : $this->client->getSite();
        $subSite = !empty($options["subsite"]) ? $options["subsite"] : $this->client->getSubSite();

        $this->client->buildClient($endpoint, $version, $site, $subSite);

        return $this->client;
    }
}
