<?php

namespace Comitium5\ApiClientBundle\Cache;

use Predis\Client as RedisClient;
use Predis\Response\Status;

/**
 * Class CSRedisWrapper
 *
 * Manage the redis session
 *
 * @link http://redis.io/
 * @link https://github.com/nrk/predis
 * @link https://github.com/snc/SncRedisBundle
 *
 * @author Oscar Jimenez <oscarjg19.developer@gmail.com>
 * @package ComitiumSuite\Bundle\CSCoreBundle\Wrapper\Cache
 */
class RedisWrapper implements MemoryCacheInterface
{
    /**
     * The code when a request is correct
     *
     * @name RESPONSE_OK
     */
    const RESPONSE_OK = 'OK';

    /**
     * @var Client $client The redis client
     * @access private
     */
    private $client;

    /**
     * @var string|array $host the host of the redis client
     * @access public
     */
    public $host;

    /**
     * @var string|array $scheme the type of connection
     * @access public
     */
    public $scheme;

    /**
     * @var string|array $port the port for the redis connection
     * @access public
     */
    public $port;

    /**
     * @var int|array
     * @access public
     */
    public $database;

    /**
     * @var string|array|null
     */
    public $password;

    /**
     * RedisWrapper constructor.
     * @param $host
     * @param $scheme
     * @param $port
     * @param $dataBase
     * @param $password
     */
    public function __construct($host, $scheme, $port, $dataBase, $password)
    {
        $this->host     = $host;
        $this->scheme   = $scheme;
        $this->port     = $port;
        $this->database = $dataBase;
        $this->password = $password;
    }

    /**
     * @return Client|RedisClient
     */
    private function client()
    {
        if (!empty($this->client)) {
            return $this->client;
        }

        $parameters = null;
        $options = null;

        if (is_array($this->host)) {
            foreach ($this->host as $i => $host) {
                $parameters[] = [
                    "host"     => $host,
                    "scheme"   => (is_array($this->scheme) ? $this->scheme[$i] : $this->scheme),
                    "port"     => (is_array($this->port) ? $this->port[$i] : $this->port),
                    "database" => (is_array($this->database) ? $this->database[$i] : $this->database),
                    "password" => (is_array($this->password) ? $this->password[$i] : $this->password),
                ];
            }

            $options = [
                'cluster' => 'predis',
            ];
        } else {
            $parameters = [
                "host"     => $this->host,
                "scheme"   => $this->scheme,
                "port"     => $this->port,
                "database" => $this->database,
                "password" => $this->password,
            ];
        }

        $this->client = new RedisClient($parameters, $options);

        return $this->client;
    }

    /**
     * Set a key to the redis database
     *
     * @access public
     *
     * @param string $key   name of the redis key
     * @param string $value value to save
     * @param int    $ttl   (optional) expiration
     * @param string $index (optional) if is a collection
     *
     * @return bool
     */
    public function set($key, $value, $ttl = null, $index = null)
    {
        $success = false;

        if ($ttl !== null) {
            $ttl = (int) $ttl === 0 ? null : (int) $ttl;
        }

        if ($this->isConnected()) {
            $value = serialize($value);

            if ($index) {
                $response = $this
                    ->client()
                    ->hset($key, $index, $value);
            } else {
                $response = $this
                    ->client()
                    ->set($key, $value);
            }

            $success = $this->checkResponse($response);

            if ($success) {
                if ($ttl !== null) {
                    $success = $this->setTTL($key, $ttl);
                }
            }
        }

        return $success;
    }

    /**
     * Get a key from the redis database
     *
     * @access public
     *
     * @param string $key   the key name
     * @param string $index (optional) if the key pertain to a index
     *
     * @return mixed|null
     */
    public function get($key, $index = null)
    {
        $response = null;

        if ($this->isConnected()) {
            if ($index) {
                $response = $this
                    ->client()
                    ->hget($key, $index);
            } else {
                $response = $this
                    ->client()
                    ->get($key);
            }
        }

        if (is_string($response)) {
            $response = unserialize($response);
        }

        return $response;
    }

    /**
     * Delete a key from redis
     *
     * @access public
     *
     * @param string $key   the key name
     * @param string $index (optional) if the key pertain to a index
     *
     * @return bool
     */
    public function delete($key, $index = null)
    {
        $success = false;

        if ($this->isConnected()) {
            if ($index) {
                $result = $this
                    ->client()
                    ->hdel($key, array($index));
            } else {
                $result = $this
                    ->client()
                    ->del(array($key));
            }

            $success = ($result > 0) ? true : false;
        }

        return $success;
    }

    /**
     * Set a expiration to the key
     *
     * @access public
     *
     * @param string $key the key name
     * @param int    $ttl the time in seconds to expire
     *
     * @return bool
     */
    public function setTTL($key, $ttl)
    {
        $success = false;

        if ($this->isConnected()) {
            $success = $this
                ->client()
                ->expire($key, $ttl);
        }

        return $success;
    }

    /**
     * Check if the given key exists
     *
     * @access public
     *
     * @param string $key   the key name
     * @param string $index (optional) if the key pertain to a index
     *
     * @return int|mixed
     */
    public function exists($key, $index = null)
    {
        $success = false;

        if ($this->isConnected()) {
            if ($index) {
                $success = $this
                    ->client()
                    ->hexists($key, $index);
            } else {
                $success = $this
                    ->client()
                    ->exists($key);
            }
        }

        return $success;
    }

    /**
     * Check if the response is ok/ko
     *
     * @access public
     *
     * @param mixed $response the response of the operation
     *
     * @return bool
     */
    protected function checkResponse($response)
    {
        $success = $response && ($response instanceof Status && self::RESPONSE_OK === $response->getPayload());

        return $success;
    }

    /**
     * Check if the client is connected, and if not, connect
     *
     * @access public
     *
     * @return bool
     */
    private function isConnected()
    {
        $success = $this
            ->client()
            ->isConnected();

        if (!$success) {
            $this
                ->client()
                ->connect();
            $success = $this
                ->client()
                ->isConnected();
        }

        return $success;
    }
}
