<?php

namespace Comitium5\ApiClientBundle\ApiClient;

use Comitium5\ApiClientBundle\Cache\MemoryCacheInterface;
use Comitium5\ApiClientBundle\Generator\SignatureGeneratorInterface;
use Comitium5\ApiClientBundle\Generator\TokenGeneratorInterface;
use Comitium5\ApiClientBundle\Parser\ResponseParser;
use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\UriTemplate\UriTemplate;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response as ResponseSymfony;

/**
 * Class Client
 *
 * @deprecated
 * @package Comitium5\ApiClientBundle\ApiClient
 */
class Client implements ClientInterface
{
    const SIGNATURE_HEADER = 'X-API-CS-Signature';

    const TOKEN_HEADER = 'X-API-CS-Token';

    const CACHE_KEY_LOCKED = '%s-locked';

    const ACCEPT_LANGUAGE_HEADER = 'Accept-Language';

    const FORCE_GET_CONTENT_HEADER = 'X-API-CS-FORCE-CONTENT';

    const CACHE_KEYS_KEY = 'entity_keys_%s_%s';

    const TIMEOUT = 7;

    /**
     * @var boolean
     */
    private $forceGetContent;

    /**
     * @var GuzzleClient
     */
    private $client;

    /**
     * @var SignatureGeneratorInterface
     */
    private $signatureGenerator;

    /**
     * @var TokenGeneratorInterface
     */
    private $tokenGenerator;

    /**
     * @var array
     */
    private $requestCache = [];

    /**
     * @var MemoryCacheInterface
     */
    private $cacheDriver;

    /**
     * @var array
     */
    private $cacheHeaders;

    /**
     * @var null
     */
    private $cacheTTL;

    /**
     * @var bool
     */
    private $useCache;

    /**
     * @var string
     */
    private $localeCode;

    /**
     * @var
     */
    private $endPoint;
    /**
     * @var
     */
    private $version;
    /**
     * @var
     */
    private $site;
    /**
     * @var
     */
    private $subSite;

    /**
     * Client constructor.
     * @param $endpoint
     * @param array $options
     * @param bool|false $useCache
     * @param null $cacheTTL
     * @param array $cacheHeaders
     */
    public function __construct(
        $endpoint,
        array $options = [],
        $useCache = false,
        $cacheTTL = null,
        array $cacheHeaders = []
    ) {
        $this->useCache = $useCache;
        $this->cacheHeaders = $cacheHeaders;
        $this->cacheTTL = $cacheTTL;

        $this->endPoint = $endpoint;

        if (!empty($options["version"])) {
            $this->version = $options["version"];
        }

        if (!empty($options["site"])) {
            $this->site = $options["site"];
        }

        if (!empty($options["subsite"])) {
            $this->subSite = $options["subsite"];
        }

        $this->buildClient();
    }

    /**
     * @param  Request $request
     * @return string
     */
    protected function generateCacheKey(Request $request)
    {
        $id = $request->attributes->get('id', '');
        if (empty($id)) {
            $id = $request->attributes->get('slug', '');
        }

        $hash = null;

        if (!empty($id) && !$request->attributes->has('related')) {
            $hash = $id;
        } else {
            $hash = $this->generateRequestHash($request);
        }

        $key = '';
        $type = $request->attributes->get('type');

        if (!empty($type)) {
            $key = $this->getEntityKey($type);
        }

        if (!empty($key) && !empty($hash)) {
            if (!empty($key)) {
                $key .= '-';
            }
            $key .= $hash;
        }

        return $key;
    }

    /**
     * @param Request $request
     *
     * @return null
     */
    public function getCacheContent(Request $request)
    {
        $content = false;

        $key = $this->generateCacheKey($request);

        if (!empty($key)) {
            $content = $this->checkKey($key);

            if (empty($content)) {
                $content = false;
            }
        }

        return $content;
    }

    /**
     * @param Request $request
     * @param $content
     */
    public function setCacheContent(Request $request, $content)
    {
        $key = $this->generateCacheKey($request);

        if (!empty($key)) {
            $this->cacheDriver->set($key, $content, 120);
            $this->unlockKey($key);
        }
    }

    /**
     * @param $key
     * @param null $index
     */
    protected function checkKey($key, $index = null)
    {
        $currentTime = time();
        $rejected    = false;

        $lockedKey = $key;

        if (!empty($index)) {
            $lockedKey .= '-'.$index;
        }
        $lockedKey = sprintf(self::CACHE_KEY_LOCKED, $lockedKey);

        $content = $this->cacheDriver->get($key, $index);

        while (
            $this->cacheDriver->get($lockedKey)
            && empty($content)
            && !$rejected
        ) {
            if ((time()-$currentTime <= self::TIMEOUT)) {
                usleep(100000); // 0.1 seconds
                $content = $this->cacheDriver->get($key, $index);
            } else {
                $rejected = true;
            }
        }

        if ($rejected) {
            $content = '';
        } elseif (empty($content)) {
            $this->cacheDriver->set($lockedKey, true, 5);
        }

        return $content;
    }

    /**
     * @param $key
     * @param null $index
     */
    protected function unlockKey($key, $index = null)
    {
        if (!empty($index)) {
            $key .= '-'.$index;
        }
        $key = sprintf(self::CACHE_KEY_LOCKED, $key);

        $this->cacheDriver->delete($key);
    }

    /**
     * @param  Request $request
     * @return string
     */
    protected function generateRequestHash(Request $request)
    {
        $id  = serialize($request->getMethod());
        $id .= ' ' . serialize($request->request->all());
        $id .= ' ' . serialize($request->query->all());

        $allAttributes = $request->attributes->all();

        if (!empty($allAttributes["_route_params"])) {
            $id .= ' '. serialize($allAttributes["_route_params"]);
        }

        return hash('md5', $id);
    }

    /**
     * @param $method
     * @param $headers
     * @return bool
     */
    public function isRequestCacheable($method, $headers)
    {
        if (!$this->useCache || !in_array(strtoupper($method), [Request::METHOD_GET, Request::METHOD_HEAD])) {
            return false;
        }

        $private_headers = array('AUTHORIZATION', 'COOKIE');

        foreach ($headers as $header => $headerValue) {
            if (array_key_exists(strtoupper($header), $private_headers)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param $method
     * @param  array $url
     * @param  array $options
     * @return mixed
     */
    private function sendRequest($method, array $url, array $options = [])
    {
        $parameters = array();

        if (isset($options['query'])) {
            $parameters = $options['query'];
            array_walk(
                $parameters,
                array($this, 'valuesToString')
            );
        }

        $attributes = $url[1];

        if (empty($attributes)) {
            $attributes = $this->getAttributesFromUrl($url[0]);
        }

        $httpRequest = new Request(
            $parameters,
            [],
            $attributes
        );

        $httpRequest->setMethod($method);

        $content = false;

        $isRequestCacheable = $this->isRequestCacheable($method, $options);

        if ($isRequestCacheable) {
            $content = $this->getCacheContent($httpRequest);
        }

        if ($content === false) {
            try {
                $options = $this->buildOptions($options);
                $parse = new UriTemplate();
                list($template, $variables) = $url;
                $url = $parse->expand($template, $variables);

                $response = $this
                    ->client
                    ->request($method, $url, $options);

                if (
                    $isRequestCacheable &&
                    !in_array($response->getStatusCode(), [
                        ResponseSymfony::HTTP_INTERNAL_SERVER_ERROR,
                        ResponseSymfony::HTTP_BAD_GATEWAY,
                        ResponseSymfony::HTTP_SERVICE_UNAVAILABLE,
                        ResponseSymfony::HTTP_GATEWAY_TIMEOUT,
                    ])
                ) {
                    $response->withAddedHeader('X-CACHE', 'MISS');
                }

                if ($this->isEntity($httpRequest) === false) {

                    $content = (string) $response->getBody();

                    if (!empty($content)) {
                        $this->setCacheContent($httpRequest, $content);
                    }

                    $response->getBody()->seek(0);
                }
            } catch (RequestException $e) {
                if ($e->hasResponse()) {
                    $response = $e->getResponse();
                } else {
                    $response = new Response(ResponseSymfony::HTTP_INTERNAL_SERVER_ERROR);
                }
            }
        } else {
            $code = empty($content) ? ResponseSymfony::HTTP_NO_CONTENT : ResponseSymfony::HTTP_NOT_MODIFIED;

            $response = new Response(
                $code,
                array(
                    'Content-Type' => 'application/json',
                    'X-CACHE' => 'HIT',
                ),
                $content
            );
        }

        return $response;
    }

    /**
     * @param $value
     */
    protected function valuesToString(&$value)
    {
        if (!is_array($value)) {
            $value = trim($value);
        } else {
            array_walk(
                $value,
                array($this, 'valuesToString')
            );
        }
    }

    /**
     * @deprecated
     * @param $requestId
     * @param $response
     * @return mixed
     */
    public function saveCache($requestId, Response $response)
    {
        $gResponse = array(
            'status_code'   => $response->getStatusCode(),
            'headers'       => $response->getHeaders(),
            'stream'        => $response->getBody()->__toString(),
        );

        $this->requestCache[$requestId] = $gResponse;

        if ($this->cacheTTL == 'rand') {
            $ttl = mt_rand(60, 120);
        } else {
            $ttl = (int) $this->cacheTTL;
        }

        return $this->cacheDriver->set($requestId, $gResponse, $ttl <= 0 ? null : $ttl);
    }

    /**
     * @deprecated
     * @param $requestId
     * @return mixed
     */
    public function resolveCache($requestId)
    {
        if (isset($this->requestCache[$requestId])) {
            return $this->requestCache[$requestId];
        }

        return $this->cacheDriver->get($requestId);
    }

    /**
     * @deprecated
     * @param  array    $gResponse
     * @return Response
     */
    public function buildResponse(array $gResponse)
    {
        $response = new Response(
            $gResponse['status_code'],
            $gResponse['headers'],
            $gResponse['stream']);

        return $response;
    }

    /**
     * {@inheritdoc}
     */
    public function call($method, $url, array $parameters = array(), array $request_config = array())
    {
        return $this->sendRequest($method, array($url, $parameters), $request_config);
    }

    /**
     * {@inheritdoc}
     */
    public function get($url, array $parameters = array(), array $request_config = array())
    {
        return $this->call('GET', $url, $parameters, $request_config);
    }

    /**
     * {@inheritdoc}
     */
    public function put($url, array $parameters = array(), array $request_config = array())
    {
        return $this->call('PUT', $url, $parameters, $request_config);
    }

    /**
     * {@inheritdoc}
     */
    public function post($url, array $parameters = array(), array $request_config = array())
    {
        return $this->call('POST', $url, $parameters, $request_config);
    }

    /**
     * {@inheritdoc}
     */
    public function patch($url, array $parameters = array(), array $request_config = array())
    {
        return $this->call('PATCH', $url, $parameters, $request_config);
    }

    /**
     * {@inheritdoc}
     */
    public function delete($url, array $parameters = array(), array $request_config = array())
    {
        return $this->call('DELETE', $url, $parameters, $request_config);
    }

    /**
     * @param Response $response
     *
     * @return bool
     */
    public function isJSON(Response $response)
    {
        return 'application/json' === $response->getHeader('Content-Type');
    }

    /**
     * @param $url
     *
     * @return array
     */
    protected function getAttributesFromUrl($url)
    {
        $result = array();

        if (!empty($url)) {
            $data = explode('/', $url);
            if (count($data) > 0) {
                $result['type'] = $data[0];
                $defineId = isset($data[1]);
                if ($result['type'] === ResourcesTypes::DIRECTORY) {
                    if (isset($data[2]) && $data[2] != 'related') {
                        $result['type'] = 'directories.items';
                        if ($data[1] == 'items') {
                            if (is_numeric($data[2])) {
                                $result['id'] = $data[2];
                            } else {
                                $result['slug'] = $data[2];
                            }
                            if (isset($data[3]) && $data[3] == 'related') {
                                $result['related'] = true;
                            }
                        } else {
                            $result['directory'] = $data[1];
                            if (isset($data[3])) {
                                if (is_numeric($data[3])) {
                                    $result['id'] = $data[3];
                                } else {
                                    $result['slug'] = $data[3];
                                }
                            }
                        }
                        $defineId = false;
                    }
                } elseif ($result['type'] === ResourcesTypes::GALLERIES) {
                    if (isset($data[2]) && $data[2] != 'related') {
                        $result['type'] = 'galleries.assets';
                        $result['gallery'] = $data[1];
                        if (isset($data[3])) {
                            if (is_numeric($data[3])) {
                                $result['id'] = $data[3];
                            } else {
                                $result['slug'] = $data[3];
                            }
                        }
                    }
                } elseif ($result['type'] === ResourcesTypes::LIVE_EVENT) {
                    if (isset($data[2]) && $data[2] === 'comments') {
                        $result['type'] = "liveEvents.comments";
                    }
                }

                if (isset($data[2]) && $data[2] == 'related') {
                    $result['related'] = true;
                }

                if ($defineId && ($result['type'] != ResourcesTypes::COMMENT || $data[1] != 'total')) {
                    if (is_numeric($data[1])) {
                        $result['id'] = $data[1];
                    } else {
                        $result['slug'] = $data[1];
                    }
                }
            }
        }

        return $result;
    }

    /**
     * @return boolean
     */
    public function isCacheHeaders()
    {
        return count($this->cacheHeaders) > 0;
    }

    /**
     * @return int
     */
    public function getCacheTTL()
    {
        return $this->cacheTTL;
    }

    /**
     * @return boolean
     */
    public function isUseCache()
    {
        return $this->useCache;
    }

    /**
     * @param array $cacheHeaders
     *
     * @return $this
     */
    public function setCacheHeaders(array $cacheHeaders)
    {
        $this->cacheHeaders = $cacheHeaders;

        return $this;
    }

    /**
     * @return array
     */
    public function getCacheHeaders()
    {
        return $this->cacheHeaders;
    }

    /**
     * @param $cacheTTL
     * @return $this
     */
    public function setCacheTTL($cacheTTL)
    {
        $this->cacheTTL = $cacheTTL;

        return $this;
    }

    /**
     * @param $useCache
     * @return $this
     */
    public function setUseCache($useCache)
    {
        $this->useCache = $useCache;

        return $this;
    }

    /**
     * @param  SignatureGeneratorInterface $signatureGenerator
     * @return $this
     */
    public function setSignatureGenerator(SignatureGeneratorInterface $signatureGenerator)
    {
        $this->signatureGenerator = $signatureGenerator;

        return $this;
    }

    /**
     * @param  TokenGeneratorInterface $tokenGenerator
     * @return $this
     */
    public function setTokenGenerator(TokenGeneratorInterface $tokenGenerator)
    {
        $this->tokenGenerator = $tokenGenerator;

        return $this;
    }

    /**
     * @param MemoryCacheInterface $cacheDriver
     *
     * @return $this
     */
    public function setCacheDriver(MemoryCacheInterface $cacheDriver)
    {
        $this->cacheDriver = $cacheDriver;

        return $this;
    }

    /**
     * @param $code
     *
     * @return mixed
     */
    public function setAcceptLanguage($code)
    {
        $this->localeCode = $code;
        $this->cacheHeaders[self::ACCEPT_LANGUAGE_HEADER] = $code;
    }

    /**
     * @param array $options
     *
     * @return array
     */
    private function buildHeaders($options = [])
    {
        $salt = md5(mt_rand(0, mt_rand()).microtime(true));

        $headers = $this->cacheHeaders;

        $body = isset($options['body']) ? $options['body'] : '';

        $headers[self::TOKEN_HEADER] = $this->tokenGenerator->generateToken($salt);
        $headers[self::SIGNATURE_HEADER] =  $this->signatureGenerator->generateSignature($body, $salt);
        $headers["Accept"] =  "application/json";

        return $headers;
    }

    /**
     * @param array $options
     *
     * @return array
     */
    private function buildOptions($options = [])
    {
        $options = array_merge(
            array(
                'body' => '',
                'headers' => [],
            ),
            $options
        );

        $options['headers'] = array_merge($this->buildHeaders($options), $options['headers']);

        return $options;
    }

    /**
     * @param bool $save
     */
    private function buildCacheKeys($save = true)
    {
        $keys = $this->get("keys", $this->buildOptions());

        if ($keys instanceof Response) {
            $keys = ResponseParser::decodeResponse($keys);

            if ($save === true) {
                $this->cacheDriver->set(sprintf(self::CACHE_KEYS_KEY, $this->subSite, $this->localeCode), $keys);
            }

            return $keys;
        }

        return;
    }

    /**
     * @param $type
     *
     * @return string
     */
    protected function getEntityKey($type)
    {
        if (preg_match('/directories(\/\d+)?\/items/i', $type)) {
            $type = "directories.items";
        }

        if (preg_match('/galleries\/\d+\/assets/i', $type)) {
            $type = "galleries.assets";
        }

        if (preg_match('/live-events\/\d+\/comments/i', $type)) {
            $type = "liveEvents.comments";
        }

        if (preg_match('/live-events/i', $type)) {
            $type = "liveEvents";
        }

        $keys = $this->cacheDriver->get(sprintf(self::CACHE_KEYS_KEY, $this->subSite, $this->localeCode));

        if (!$keys) {
            // Get keys from API
            $keys = $this->buildCacheKeys();
        }

        $allKeys = $keys['keys'];

        // Check if there is a new key
        if (!isset($allKeys[$type])) {
            $keys = $this->buildCacheKeys();
        }

        $allKeys = $keys['keys'];

        return isset($allKeys[$type]) ? $keys["prefix"].$allKeys[$type] : null;
    }

    /**
     * @return mixed
     */
    public function getLocaleCode()
    {
        return $this->localeCode;
    }

    /**
     * @param $localeCode
     * @return $this
     */
    public function setLocaleCode($localeCode)
    {
        $this->localeCode = $localeCode;

        return $this;
    }

    /**
     * @param $forceGetContent
     * @return $this
     */
    public function setForceGetContent($forceGetContent)
    {
        $this->forceGetContent = $forceGetContent;
        $this->cacheHeaders[self::FORCE_GET_CONTENT_HEADER] = $forceGetContent;

        return $this;
    }

    /**
     * @return GuzzleClient
     */
    public function getClient()
    {
        return $this->client;
    }

    /**
     * @param GuzzleClient $client
     * @return $this
     */
    public function setClient(GuzzleClient $client)
    {
        $this->client = $client;
        return $this;
    }

    /**
     * @return array
     */
    public function getRequestCache()
    {
        return $this->requestCache;
    }

    /**
     * @param array $requestCache
     * @return $this
     */
    public function setRequestCache($requestCache)
    {
        $this->requestCache = $requestCache;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getEndPoint()
    {
        return $this->endPoint;
    }

    /**
     * @param mixed $endPoint
     * @return $this
     */
    public function setEndPoint($endPoint)
    {
        $this->endPoint = $endPoint;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getVersion()
    {
        return $this->version;
    }

    /**
     * @param mixed $version
     * @return $this
     */
    public function setVersion($version)
    {
        $this->version = $version;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getSite()
    {
        return $this->site;
    }

    /**
     * @param mixed $site
     * @return $this
     */
    public function setSite($site)
    {
        $this->site = $site;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getSubSite()
    {
        return $this->subSite;
    }

    /**
     * @param mixed $subSite
     * @return $this
     */
    public function setSubSite($subSite)
    {
        $this->subSite = $subSite;
        return $this;
    }

    /**
     * @param null $endpoint
     * @param null $version
     * @param null $site
     * @param null $subSite
     * @param bool $verify
     * @param bool $httpErrors
     */
    public function buildClient(
        $endpoint = null,
        $version = null,
        $site = null,
        $subSite = null,
        $verify = false,
        $httpErrors = true
    ) {
        $endpoint = $endpoint ?: $this->endPoint;

        $options = array(
            "version" => $version ? $version : $this->version,
            "site"    => $site ? : $this->site,
            "subsite" => $subSite ? : $this->subSite,
        );

        $this->version = $options["version"];
        $this->site = $options["site"];
        $this->subSite = $options["subsite"];

        $template = new UriTemplate();
        $uri = $template->expand($endpoint, $options);

        $this->client = new GuzzleClient([
            'base_uri'    => $uri,
            'headers'     => [
                'Accept' => 'application/json',
            ],
            'verify'      => $verify,
            'http_errors' => $httpErrors,
        ]);
    }

    /**
     * @param $httpRequest
     * @return bool
     */
    private function isEntity(Request $httpRequest)
    {
        if (
            (
                $httpRequest->attributes->get('id', '') != '' ||
                $httpRequest->attributes->get('slug', '') != ''
            ) && !$httpRequest->attributes->has('related')
        ) {
            return true;
        }

        return false;
    }
}
